// ArchiveCommand.cs
//
// Holds all information a archive command needs
//
// Copyright (C) 2004 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>


using System;
using System.Collections;
using System.IO;
using System.Xml;

namespace Upkg
{
	class ArchiveCommand : Command
	{
		// holds the archive name
		public readonly ResolvableString Name;
		// holds the type - currently only tar suportet
		public readonly string Type = "tar";
		// holds the compression
		public readonly string Compression;
		// shall the dir created by the archive be deleted uppon all commands are finished
		public readonly bool Delete;
		// dir where the source shall be unpacked to
		public readonly ResolvableString SrcDir;
		// This is the place, if set, where all the commands inside an archive tag will be executed
		public readonly ResolvableString BuildDir;
		// how many trailing slashes shall be stripped during extraction?
		public readonly uint Strip;
		// commands inside the archive element
		public readonly ArchiveScript Script;
		// the archive file
		public override FileList Sources { get { return (sources); }}
		
		// constructor - simple variant
		public ArchiveCommand (string name, ArchiveScript script) : this (name, "tar", Global.Compression, true, null, null, 1, script)
		{
		}
		
		// constructor
		public ArchiveCommand (string name, string type, string compression, bool delete, string srcdir, string builddir, uint strip, ArchiveScript script)
		{
			// check the arguments
			if (name == null)
				throw new ArgumentNullException ("name");
			if (type == null)
				throw new ArgumentNullException ("type");
			if (type != "tar" && type != "tgz")
				throw new NotImplementedException ("ArchiveCommand: currently only tar and tgz archives are implemented");
			if (compression == null)
				throw new ArgumentNullException ("compression");
			if (compression != "plain" && compression != "gz" && compression != "bz2" && compression != "xz" && compression != "zst")
				throw new NotImplementedException ("ArchiveCommand: not supportet compression \"" + compression + "\" requested");
			
			// assign the values
			Name = name;
			Compression = compression;
			Delete = delete;
			Type = type;
			if (srcdir == null)
				SrcDir = Path.GetFileName (Name);
			else
				SrcDir = srcdir;
			BuildDir = builddir;
			Strip = strip;
			filename = name + "." + Type;
			if (Compression != "plain")
				filename += "." + Compression;
			sources.Add (filename);
			if (script != null)
			{
				Script = script;
				// add the child sources
				Sources.AddRange (Script.Sources);
			}
		}
		
		// constructor with XmlElement
		public ArchiveCommand (XmlElement archivetype)
		{
			if (archivetype == null)
				throw new ArgumentNullException ("archivetype");
			
			// Get the name
			string name = archivetype.Attributes.GetNamedItem("name").InnerText;
			Name = name;
			// Get the type
			if (archivetype.Attributes.GetNamedItem("type") != null)
			{
				string type = archivetype.Attributes.GetNamedItem("type").InnerText;
				if (type != "tar" && type != "tgz")
					throw new NotImplementedException ("ArchiveCommand: currently only tar and tgz archives are implemented");
				else
					Type = type;
			}
			else
				Type = "tar";
			// Get the compression
			if (archivetype.Attributes.GetNamedItem("compression") != null)
				Compression = archivetype.Attributes.GetNamedItem("compression").InnerText;
			else
			{
				// check for implied compressions
				if (Type == "tgz")
					Compression = "plain";
				else
					Compression = Global.Compression;
			}
			// Get the srcdir
			SrcDir = archivetype.GetAttribute ("srcdir");
			if (SrcDir == "")
				SrcDir = Path.GetFileName (name);
			// Get the stip val
			if (archivetype.Attributes.GetNamedItem("strip") != null)
				Strip = uint.Parse (archivetype.Attributes.GetNamedItem("strip").InnerText);
			else
				Strip = 1;
			// Get the builddir
			BuildDir = archivetype.GetAttribute ("builddir");
			if (BuildDir == "")
				BuildDir = null;
			// Check if we want to delete the build dir after using it
			if ((archivetype.Attributes.GetNamedItem("delete") != null) && (archivetype.Attributes.GetNamedItem("delete").InnerText == "no"))
				Delete = false;
			else
				Delete = true;
			// Get the filename
			filename = name + "." + Type;
			if (Compression != "plain")
				filename += "." + Compression;
			// Add the file to the filelist
			sources.Add (filename);
			
			// build the child script if necessary
			if (archivetype.ChildNodes.Count > 0)
			{
				Script = new ArchiveScript (SrcDir);
				Script.Process(archivetype);
				// add the child sources
				Sources.AddRange (Script.Sources);
			}
		}
		
		// overriden shell script func
		public override BashCommand ToBashCommand (bool verbose)
		{
			BashList result = new BashList ();
			
			// remove build and srcdirs
			if (BuildDir != null)
				result.Add (new SimpleCommand("rm -rf " + BuildDir, true).ToBashCommand (verbose));
			result.Add (new SimpleCommand("rm -rf " + SrcDir, true).ToBashCommand (verbose));
			
			//Create the srcdir and switch to it
			result.Add (new SimpleCommand("mkdir " + SrcDir, true).ToBashCommand (verbose));
			result.Add (new SimpleCommand("pushd " + SrcDir, true).ToBashCommand (verbose));
			
			// treat the gathered information
			result.Add (new SimpleCommand ("tar --no-same-owner --strip-components=" + Strip + " -xf " + "$PACKAGESOURCEDIR/" + Path.GetFileName (filename), true).ToBashCommand (verbose));
			
			// check if we want a builddir
			if (BuildDir != null)
			{
				result.Add (new SimpleCommand("popd", true).ToBashCommand (verbose));
				result.Add (new SimpleCommand("mkdir " + BuildDir, true).ToBashCommand (verbose));
				result.Add (new SimpleCommand("pushd " + BuildDir,true).ToBashCommand (verbose));
			}
			
			// Call ProcessScriptType recoursiveley because the archive elements is also of type scriptType
			if (Script != null)
				result.Add (Script.ToBashCommand (verbose));
			
			// Go up a dir
			result.Add (new SimpleCommand("popd",true).ToBashCommand (verbose));
			// Do we want to delete the build dir
			if (Delete)
			{
				if (BuildDir != null)
					result.Add (new SimpleCommand("rm -rf " + BuildDir,true).ToBashCommand (verbose));
				result.Add (new SimpleCommand("rm -rf " + SrcDir,true).ToBashCommand (verbose));
			}
			
			return (result);
		}
		
		private FileList sources = new FileList ();
		protected ResolvableString filename;
	}
}
