// Branch.cs
//
// Represents a branch (stable, testing, unstable or experimental)
//
// Copyright (C) 2004 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>


using System;
using System.Collections;
using System.Xml;
using System.IO;

namespace Upkg
{
	public struct Branch : IComparable
	{		
		public Branch (string branchName)
		{
			if (branchName == null)
				throw (new ArgumentNullException ("branchName"));

			value = SByte.MinValue;

			if (branchName == "stable")
				value = 0;
			else if (branchName == "testing")
				value = -1;
			else if (branchName == "unstable")
				value = -2;
			else if (branchName == "experimental")
				value = -3;
			else
				// regard an unknown branch as very unstable
				// so the value has to be -4 or lower but it has to be unique
				value = (sbyte) -(4 + Math.Abs (branchName.GetHashCode ()) % 100);
		}
		
		public int CompareTo (object obj)
		{
			// per specification we have to return a positive value when the specified object is null
			if (obj == null)
				return (1);

			if (!(obj is Branch))
				throw new ArgumentException ("Value is not a Branch");

			Branch branch = (Branch) obj;
			return (value.CompareTo (branch.value));
		}

		public static bool operator< (Branch b1, Branch b2)
		{ 
			return (b1.CompareTo (b2) < 0);
		}

		public static bool operator<= (Branch b1, Branch b2)
		{ 
			return (b1.CompareTo (b2) <= 0);
		}

		public static bool operator> (Branch b1, Branch b2)
		{ 
			return (b1.CompareTo (b2) > 0);
		}

		public static bool operator>= (Branch b1, Branch b2)
		{ 
			return (b1.CompareTo (b2) >= 0);
		}
		
		public static bool operator== (Branch b1, Branch b2)
		{ 
			return (b1.CompareTo (b2) == 0);
		}
		
		public static bool operator!= (Branch b1, Branch b2)
		{ 
			return (b1.CompareTo (b2) != 0);
		}
		
		public static Branch operator++ (Branch b)
		{ 
			b.value++;
			return (b);
		}
		
		public static Branch operator-- (Branch b)
		{ 
			b.value--;
			return (b);
		}
		
		public static implicit operator Branch (string s)
		{
			return (new Branch(s));
		}
		
		public static implicit operator string (Branch b)
		{
			return (b.ToString());
		}
		
		public override bool Equals (object obj)
		{
			if (!(obj is Branch))
				return (false);

			return (((Branch) obj).value == value);
		}
		
		public override string ToString ()
		{
			switch (value)
			{
			case 0:
				return ("stable");
			case -1:
				return ("testing");
			case -2:
				return ("unstable");
			case -3:
				return ("experimental");
			default:
				return ("unknown");
			}
		}

		public override int GetHashCode()
		{
			return (value.GetHashCode ());
		}
		
		// retuns an array with all default branches
		public static Branch[] DefaultBranches = new Branch[] { new Branch("stable"),new Branch("testing"),new Branch("unstable"),new Branch("experimental") };
		
		// value represents the stability of the branch
		private sbyte value;
	}
}
