// BranchSources.cs
//
// Holds list of needed sources for a given Branch
//
// Copyright (C) 2004-2005 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.IO;
using System.Collections;


namespace Upkg
{
	public class BranchSources
	{
		// holds all architectures the sources are collected from
		public readonly string[] Architectures;
		// holds all branches the sources are collected from
		public readonly Branch[] Branches;
		// holds all local sources
		public SortedList LocalSources { get { return (localsources); }}
		// holds all remote sources
		public SortedList RemoteSources { get { return (remotesources); }}
		// implementation of LocalSources
		private readonly SortedList localsources = new SortedList ();
		// implementation of RemoteSources
		private readonly SortedList remotesources = new SortedList ();
		
		// holds the file type switches
		public readonly bool SourceFiles;
		public readonly bool BinaryFiles;
		public readonly bool StateFiles;
		public readonly bool SpecFiles;
		
		// constructor
		// sources, binaries and state are switches whether to consider these types of files
		public BranchSources (string[] architectures, Branch[] branches, bool sourcefiles, bool binaryfiles, bool statefiles, bool specfiles)
		{
			if (architectures == null)
				throw (new ArgumentNullException ("architectures"));
			if (branches == null)
				throw (new ArgumentNullException ("branches"));
			
			Architectures = architectures;
			Branches = branches;
			SourceFiles = sourcefiles;
			BinaryFiles = binaryfiles;
			StateFiles = statefiles;
			SpecFiles = specfiles;
		}
		
		private void ProcessRelease (ReleaseSpecification release, string repopath)
		{
			release.Settings.Activate ();
			
			// add the spec file
			if (SpecFiles)
			{
				// add "normal" spec files
				if (!localsources.ContainsKey (Path.Combine (repopath, "specs/" + release.Package.Name + ".xml")))
					localsources.Add (Path.Combine (repopath, "specs/" + release.Package.Name + ".xml"), "specs/" + release.Package.Name + ".xml");
				// add "tag" spec files
				if (!localsources.ContainsKey (Path.Combine (repopath, "specs/tag-" + release.Tag + ".xml")) && File.Exists (Path.Combine (repopath, "specs/tag-" + release.Tag + ".xml")))
					localsources.Add (Path.Combine (repopath, "specs/tag-" + release.Tag + ".xml"), "specs/tag-" + release.Tag + ".xml");
				// add arch-$ARCH.xml
				if (!localsources.ContainsKey (Path.Combine (repopath, "specs/arch-" + Local.Architecture + ".xml")))
					localsources.Add (Path.Combine (repopath, "specs/arch-" + Local.Architecture + ".xml"), "specs/arch-" + Local.Architecture + ".xml");
				// add global.xml
				if (!localsources.ContainsKey (Path.Combine (repopath, "specs/global.xml")))
					localsources.Add (Path.Combine (repopath, "specs/global.xml"), "specs/global.xml");
			}				
			
			// don't do anything further if this isn't a resonable release
			if (release.Virtual)
			{
				Settings.Deactivate ();
				return;
			}
			
			// go through the sourcelist
			if (SourceFiles || !release.Settings.AllowBinaryInstall)
			{
				foreach (ResolvableString source in release.BuildSources)
				{
					if (((string) source).StartsWith ("http://") || ((string) source).StartsWith ("https://"))
					{
						if (!remotesources.ContainsKey ((string)source))
							remotesources.Add ((string)source, "sources/" + release.Package.Name + "/" + Path.GetFileName (source));
					}
					else if (!localsources.ContainsKey(Path.Combine (repopath, "sources/" + release.Package.Name + "/" + source)))
						localsources.Add (Path.Combine (repopath, "sources/" + release.Package.Name + "/" + source), "sources/" + release.Package.Name + "/" + source);
				}
			}
			// only add binaries if the release generates them
			if (BinaryFiles && release.Settings.AllowBinaryInstall)
			{
				if (!localsources.ContainsKey(Path.Combine (repopath, "binaries/" + release.BinaryName)))
					localsources.Add (Path.Combine (repopath, "binaries/" + release.BinaryName), "binaries/" + release.BinaryName);
			}
			//add state files
			if (StateFiles)
			{
				if (!localsources.ContainsKey (Path.Combine (repopath, "state/info/" + release.VersionedName + ".info")))
					localsources.Add (Path.Combine (repopath, "state/info/" + release.VersionedName + ".info"), "state/info/" + release.VersionedName + ".info");
				if (!localsources.ContainsKey (Path.Combine (repopath, "state/files/" + release.VersionedName + ".digest." + Global.Compression)))
					localsources.Add (Path.Combine (repopath, "state/files/" + release.VersionedName + ".digest." + Global.Compression), "state/files/" + release.VersionedName + ".digest." + Global.Compression);
				if (!localsources.ContainsKey (Path.Combine (repopath, "state/files/" + release.VersionedName + ".dirs." + Global.Compression)))
					localsources.Add (Path.Combine (repopath, "state/files/" + release.VersionedName + ".dirs." + Global.Compression), "state/files/" + release.VersionedName + ".dirs." + Global.Compression);
				if (!localsources.ContainsKey (Path.Combine (repopath, "state/files/" + release.VersionedName + ".links." + Global.Compression)))
					localsources.Add (Path.Combine (repopath, "state/files/" + release.VersionedName + ".links." + Global.Compression), "state/files/" + release.VersionedName + ".links." + Global.Compression);
				if (!localsources.ContainsKey (Path.Combine (repopath, "state/files/" + release.VersionedName + ".config." + Global.Compression)))
					localsources.Add (Path.Combine (repopath, "state/files/" + release.VersionedName + ".config." + Global.Compression), "state/files/" + release.VersionedName + ".config." + Global.Compression);
				if (!localsources.ContainsKey (Path.Combine (repopath, "state/logs/" + release.VersionedName + ".log." + Global.Compression)))
					localsources.Add (Path.Combine (repopath, "state/logs/" + release.VersionedName + ".log." + Global.Compression), "state/logs/" + release.VersionedName + ".log." + Global.Compression);
			}				
			
			Settings.Deactivate ();
		}
		
		// crawls a repo for all sources mentioned in the specified branch
		public void Crawl (string repopath)
		{
			// buffer pkg
			Package p;
			
			if (repopath == null)
				throw (new ArgumentNullException ("repopath"));
			
			// remove trailing DirectorySeparatorChars	
			repopath = repopath.TrimEnd (new char[] { Path.DirectorySeparatorChar });
			
			// go through all architectures
			foreach (string arch in Architectures)
			{
				Local.Architecture = arch;
				Local.HostArchitecture = arch;
				Global.Init ();
				
				// go through all branches
				foreach (Branch branch in Branches)
				{
					Local.Branch = branch;
					Architecture.Flush ();
					Tag.Flush ();
					Package.Flush ();
					ReleaseSpecification.ReleaseList.Clear();
					
					// Check if the branch exists
					if (Global.Branches.Current == null)
					{
						if (Local.Verbose)
							Console.Error.WriteLine ("--- Omitting branch '" + branch + "': Branch has no valid Settings");
						return;
					}
					
					// go through all files inside repopath
					foreach (string file in Directory.GetFiles (Path.Combine (repopath, "specs"), "*.xml"))
					{
						// exclude global.xml, tag-*.xml, and arch-*.xml
						if (Path.GetFileName (file) == "global.xml" || Path.GetFileName (file).StartsWith ("tag-") || Path.GetFileName (file).StartsWith ("arch-"))
							continue;
						
						// create a Package for each xml file
						p = Package.GetPackage (Path.GetFileName (file.Replace (".xml", "")));
						// add all releases of our branch to the dependency list
						foreach (Branches b in p.Tags.Values)
						{
							// b.Current processes the release
							if (b.Current != null)
								ProcessRelease ((ReleaseSpecification) b.Current, repopath);
						}
					}
				}
			}
		}
		
		// crawls a release and all its dependencies for all sources mentioned in the specified branch
		public void CrawlRelease (string package, string tag, string repopath)
		{
			// buffer pkg
			Package p;
			
			if (package == null)
				throw (new ArgumentNullException ("package"));
			if (tag == null)
				throw (new ArgumentNullException ("tag"));
			if (repopath == null)
				throw (new ArgumentNullException ("repopath"));
			
			// remove trailing DirectorySeparatorChars	
			repopath = repopath.TrimEnd (new char[] { Path.DirectorySeparatorChar });
			
			// go through all architectures
			foreach (string arch in Architectures)
			{
				Local.Architecture = arch;
				Local.HostArchitecture = arch;
				Global.Init ();
				
				// go through all branches
				foreach (Branch branch in Branches)
				{
					Local.Branch = branch;
					Package.Flush ();
					ReleaseSpecification.ReleaseList.Clear();
					
					// Check if the branch exists
					if (Global.Branches.Current == null)
					{
						if (Local.Verbose)
							Console.Error.WriteLine ("--- Omitting branch '" + branch + "': Branch has no valid Settings");
						return;
					}
					
					p = Package.GetPackage (package);
					Branches b = (Branches) p.Tags[tag];
					if (b.Current != null)
					{
						foreach (ReleaseSpecification release in ReleaseSpecification.ReleaseList.Values)
						{
							ProcessRelease (release, repopath);
						}
					}
				}
			}
		}
	}
}
