// BranchSpecification.cs
//
// Process a <branch> element
//
// Copyright (C) 2004-2005 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>


using System;
using System.Collections;
using System.Xml;
using System.IO;

namespace Upkg
{
	public class BranchSpecification : Specification
	{
		// process the <branch> element
		public BranchSpecification (XmlElement SpecElement) : this (SpecElement, null, null)
		{
		}
		
		// process the <branch> element
		public BranchSpecification (XmlElement SpecElement, string arch, string tag)
		{
			if (SpecElement == null)
				throw (new ArgumentNullException ("SpecElement"));

			this.SpecElement = SpecElement;
			this.arch = arch;
			this.tag = tag;
			Name = SpecElement.GetAttribute ("name");
			if (Name == "") {
				Name = Local.Branch.ToString ();
			}
		}
		
		// finish processing the xmlelement
		// will be called by Branches class
		public override void Process ()
		{
			if (!processed)
			{
				// mark as processed before anything else so we don't end in an endless loop
				processed = true;

				// get the BranchSpecification of the parent scope, if there is one
				BranchSpecification parentSpec = null;
				if (tag != null)
					parentSpec = (BranchSpecification) Upkg.Architecture.GetArchitecture (Local.Architecture).Branches.Current;
				if (parentSpec == null && (arch != null || tag != null))
					parentSpec = (BranchSpecification) Global.Branches.Current;
					
				// process the settings element separately as the other elements may depend on this
				Settings = ProcessSettings (parentSpec);

				if (arch == null && tag == null)
				{
					// Add branch info to the variables
					Settings.Variables.Add ("$BRANCH", Local.Branch.ToString ());
				}

				if (parentSpec != null)
				{
					Init.AddRange (parentSpec.Init);
					PostInst.AddRange (parentSpec.PostInst);
					PreRm.AddRange (parentSpec.PreRm);
					IgnoreFiles.AddRange (parentSpec.IgnoreFiles);
					Strip.AddRange (parentSpec.Strip);
					Finalize.AddRange (parentSpec.Finalize);
				}

				foreach (XmlNode node in SpecElement.ChildNodes)
				{
					XmlElement element = node as XmlElement;
					// ignore non-element nodes (as for example comments)
					if (element == null)
						continue;

					// process the childs of <branch>
					switch (element.Name)
					{
					case "init":
						Init.Process (element);
						break;
					case "ignore":
						IgnoreFiles.Process (element);
						break;
					case "postinst":
						PostInst.Process (element);
						break;
					case "prerm":
						PreRm.Process (element);
						break;
					case "strip":
						Strip.Process (element);
						break;
					case "finalize":
						Finalize.Process (element);
						break;
					}
				}
			}
		}

		public readonly string Name;
		public Settings Settings;
		// commands to be executed right before package build/installation
		public readonly Script Init = new Script ();
		// files and directories to be ignored in the package digest
		public readonly FileList IgnoreFiles = new FileList ();
		// commands to be executed after package build/installation
		public readonly Script PostInst = new Script ();
		// commands to be executed before package removal
		public readonly Script PreRm = new Script ();
		// commands to be executed before computing checksums
		public readonly Script Strip = new Script ();
		// commands to be executed after build / installation / upgrade / removal
		public readonly Script Finalize = new Script ();
		
		private bool processed = false;
		private string arch = null;
		private string tag = null;
	}
}
