// BuildScriptCollection.cs
//
// Writes one or multiple bash scripts to build and install Upkg packages
//
// Copyright (C) 2004-2006 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.IO;
using System.Collections;

namespace Upkg
{
	public class BuildScriptCollection
	{
		// filename of main script
		private readonly string filename;
		// the stream, only used when writing to stdout
		private readonly Stream stream;
		
		private readonly BashScript mainScript = new BashScript ();
		private readonly Hashtable buildScriptTable = new Hashtable ();
		
		protected ArrayList releaseList = new ArrayList ();
		protected ArrayList checkReleaseList = new ArrayList ();
	
		// default StreamWriter constructor.
		public BuildScriptCollection (string filename) : this (null, filename) { }
		
		// full StreamWriter constructor.
		// if stream is null, multiple scripts will be generated to speedup building
		public BuildScriptCollection (Stream stream, string filename)
		{
			if (filename == null)
				throw (new ArgumentNullException ("filename"));

			this.filename = filename;
			this.stream = stream;

			// write the header
			mainScript.Add (HeaderBashCommand ());

			mainScript.Add (AddReleaseFunction ());

			mainScript.Add (RepoReadonlyFunction ());
		}
		
		public void AddReleaseSpecification (ReleaseSpecification release)
		{
			// Check for non valid parameters
			if (release == null)
				throw new ArgumentNullException ("release");

			// we don't need to do anything with zombie packages
			// beside adding them to the release list to prevent
			// them from being removed
			if (!release.Zombie)
			{
				// We only need to install for real if there is a build
				// and/or a post build script (i.e. this is not a pure
				// virtual package)
				// We also need to call the install function if this
				// release is in the package selection (to create
				// the .select file)
				// post build script has only relevance for non-staging
				// packages in build mode, so only use the package if it's relevant
				if (!release.Virtual ||
					(Local.Bootstrap && !release.Settings.Staging && release.PostBuild.Count > 0) ||
					release.Selected)
				{
					release.Settings.Activate ();

					mainScript.Add (ReleaseCheckFunction (release));
					mainScript.Add (ReleaseDownloadFunction (release));
					mainScript.Add (ReleaseInstallFunction (release));
					if (!release.Virtual)
					{
						if (stream == null)
						{
							BashScript buildScript = new BashScript ();
							buildScript.Add (ReleaseBuildFunction (release));
							buildScript.Add (new BashCommand ("build_" + release.UniqueName));
							buildScriptTable[release.UniqueName] = buildScript;
						}
						else
						{
							mainScript.Add (ReleaseBuildFunction (release));
						}
					}
					
					checkReleaseList.Add (release.UniqueName);
				}
			}

			releaseList.Add (release.UniqueName);

			Settings.Deactivate ();
		}
		
		public void Write ()
		{
			mainScript.Add (FinalizeFunction ());
			mainScript.Add (MainFunction ());
			
			BashScriptWriter writer;

			if (stream == null)
				writer = new BashScriptWriter (filename);
			else
				writer = new BashScriptWriter (stream);
			
			mainScript.WriteTo (writer);
			writer.Close ();
			
			foreach (string sub in buildScriptTable.Keys)
			{
				writer = new BashScriptWriter (filename + "-" + sub);
				((BashScript) buildScriptTable[sub]).WriteTo (writer);
				writer.Close ();
			}
		}
		
		protected BashCommand HeaderBashCommand ()
		{
			string proxy;
			if (Local.Proxy != null)
			{
				proxy = "export http_proxy=" + Local.Proxy + "\n\n";
			}
			else
			{
				proxy = "";
			}
			return (new BashCommand (@"
# Automatically generated BASH script to compile Upkg packages.
# Copyright (C) 2006 Raffaele Sandrini <rasa@paldo.org>, Jürg Billeter <juerg@paldo.org>
# Modifying this script makes little sense, edit the source specs instead.
# Generated at: " + DateTime.Now.ToString () + "\n\n" + proxy + "\n\n"));
		}
		
		// writes the check function of a package: this checks whether a
		// package needs to be built from source or installed from a binary
		protected BashFunction ReleaseCheckFunction (ReleaseSpecification release)
		{
			// Check for non valid parameters
			if (release == null)
				throw new ArgumentNullException ("release");
			
			BashFunction func = new BashFunction ("check_" + release.UniqueName);

			// Write the check section

			string root = "";
			if (Local.Bootstrap && !release.Settings.Staging)
				root = release.Settings.Variables["$CHROOTDIR"] as string;
			string statedir = root + (string) release.Settings.Variables["$STATEDIR"];
		
			// Check if we need to install that package (check whether already installed and up-to-date)
			if (release.Settings.Register && !release.Virtual)
			{
				func.Add (new BashCommand ("if [ ! -f " + statedir + "/packages/" + release.VersionedName + " ]"));
				func.Add (new BashCommand ("then"));
				
				BashList checkList = new BashList ();
				checkList.Indent = true;

				// Now determine whether there is a binary available
				// or if we shall build the package from source

				string repo = release.Package.Specification.Repository.Url;

				if (Local.AllowBinaryInstall && release.Settings.AllowBinaryInstall)
				{
					checkList.Add (new BashCommand (EscapeString (release.UniqueName) + "_mode=install"));
					checkList.Add (new BashCommand (EscapeString (release.UniqueName) + "_message=\"Installing " + release.Name + " (" + Local.Branch + ")...\""));
					// First check whether we've already got the binary in the cache
					checkList.Add (new BashCommand ("[ -e " + release.Settings.Variables["$CACHEDIR"] + "/binaries/" + release.BinaryName + " ] && { add_release " + statedir + " binary " + release.UniqueName + " ; return 0 ; }"));
					// Now check whether the binary is available in the repo
					if (repo.StartsWith("http://") || repo.StartsWith("https://"))
						checkList.Add (new BashCommand ("wget --spider " + repo + "/binaries/" + release.BinaryName + " > /dev/null 2>&1 && { add_release " + statedir + " binary " + release.UniqueName + " ; return 0 ; }"));
					else
						checkList.Add (new BashCommand ("[ -e " + repo + "/binaries/" + release.BinaryName + " ] && { add_release " + statedir + " binary " + release.UniqueName + " ; return 0 ; }"));
				}
				
				// skip the package if we're in --disable-source-build mode and there could be a binary available but there is none
				if (!Local.AllowSourceBuild && release.Settings.AllowBinaryInstall)
				{
					checkList.Add (new BashCommand ("SKIP_RELEASES=\"$SKIP_RELEASES " + release.UniqueName + "\""));
				}
				else
				{
					checkList.Add (new BashCommand (EscapeString (release.UniqueName) + "_mode=build"));
					checkList.Add (new BashCommand (EscapeString (release.UniqueName) + "_message=\"Building " + release.Name + " (" + Local.Branch + ")...\""));
					// we get here if there is no binary or we don't want a binary installation
					// so let's install from source
					checkList.Add (new BashCommand ("add_release " + statedir + " source " + release.UniqueName));
				}

				// end conditional (package already installed)
				func.Add (checkList);
				func.Add (new BashCommand ("fi"));
			}
			else if (!release.Virtual ||
				(Local.Bootstrap && !release.Settings.Staging && release.PostBuild.Count > 0))
			{
				func.Add (new BashCommand (EscapeString (release.UniqueName) + "_mode=build"));
				func.Add (new BashCommand (EscapeString (release.UniqueName) + "_message=\"Updating " + release.Name + " (" + Local.Branch + ")...\""));
				// non-registered packages will just always be built
				// configuration packages and packages like kernel modules who have to be checked / updated after each upgrade
				func.Add (new BashCommand ("INSTALL_RELEASES=\"$INSTALL_RELEASES " + release.UniqueName + "\""));
				func.Add (new BashCommand ("UPDATE_RELEASES=\"$UPDATE_RELEASES " + release.UniqueName + "\""));
			}
			else
			{
				// this package is virtual (non-registered) but is in the package selection
				func.Add (new BashCommand (EscapeString (release.UniqueName) + "_message=\"Selecting " + release.Name + " (" + Local.Branch + ")...\""));
				func.Add (new BashCommand ("INSTALL_RELEASES=\"$INSTALL_RELEASES " + release.UniqueName + "\""));
			}
			
			return (func);
		}

		// writes the download function of a package: this downloads sources or binaries
		protected BashFunction ReleaseDownloadFunction (ReleaseSpecification release)
		{
			// Check for non valid parameters
			if (release == null)
				throw new ArgumentNullException ("release");

			// Write the download section
			// Function header
			BashFunction func = new BashFunction ("download_" + release.UniqueName);

			// we only need the download function
			// if we install a binary or we need some sources,
			if (!release.Virtual && ((Local.AllowBinaryInstall && release.Settings.AllowBinaryInstall) ||
						 (release.BuildSources.Count > 0 && (Local.AllowSourceBuild || !release.Settings.AllowBinaryInstall)))) {
				string umount_str;
				Repository repo = release.Package.Specification.Repository;

				// construct a valid unmount string
				umount_str = String.Empty;
				if (!repo.Remote)
					umount_str += " " + release.Settings.Variables["$CACHEDIR"];

				// add the trap
				if (umount_str != String.Empty)
					func.Add (new BashCommand ("trap \"umount -l " + umount_str + "; exit 1\" INT TERM EXIT"));
				else
					func.Add (new BashCommand ("trap \"exit 1\" INT TERM EXIT"));
		
				// Yell what we're doing
				func.Add (new BashCommand ("echo \"Downloading " + release.Name + "...\""));

				string output_redirection = "";
				if (!Local.Verbose) {
					output_redirection = "> /dev/null 2>&1";
				}

				// if this release uses sources from a local repository, don't use caching. Instead mount the repo dir into the cachedir
				if (!repo.Remote)
				{
					func.Add (new SimpleCommand ("mount --bind " + repo.Url + " " + release.Settings.Variables["$CACHEDIR"], true).ToBashCommand(Local.Verbose));
				}

				// Download binaries or sources
				func.Add (DownloadFunction (release));
				func.Add (new SimpleCommand ("download " + output_redirection, true).ToBashCommand (Local.Verbose));

				// unmount a possibly mounted places
				if (umount_str != String.Empty)
					func.Add (new SimpleCommand ("umount -l " + umount_str, false).ToBashCommand (Local.Verbose));

				// remove trap
				func.Add (new BashCommand ("trap \"\" INT TERM EXIT"));
			} else {
				func.Add (new BashCommand ("true"));
			}

			return (func);
		}

		// writes the install function of a package: this does
		// everything required to install a package, i.e., it prepares the
		// environment and calls the build function (in chroot if required)
		protected BashFunction ReleaseInstallFunction (ReleaseSpecification release)
		{
			// Check for non valid parameters
			if (release == null)
				throw new ArgumentNullException ("release");
				
			// Write the install section
			// Function header
			BashFunction func = new BashFunction ("install_" + release.UniqueName);

			string root;
			string umount_str;
			Repository repo = release.Package.Specification.Repository;
			
			if (Local.Bootstrap && !release.Settings.Staging)
				root = release.Settings.Variables["$CHROOTDIR"] as string;
			else
				root = "";
			
			// construct a valid unmount string
			umount_str = String.Empty;
			if (!release.Virtual) {
				if (Local.Bootstrap && !release.Settings.Staging) {
					if (!repo.Remote) {
						umount_str += " " + root + release.Settings.Variables["$CACHEDIR"];
					}
					umount_str += " " + root + "{" + Local.CacheDir + ",/sys,/proc,/dev}";
				}
				if (!repo.Remote)
					umount_str += " " + release.Settings.Variables["$CACHEDIR"];
			}
			
			// add the trap
			if (umount_str != String.Empty)
				func.Add (new BashCommand ("trap \"umount -l " + umount_str + "; exit 1\" INT TERM EXIT"));
			else
				func.Add (new BashCommand ("trap \"exit 1\" INT TERM EXIT"));
		
			// Yell what we're doing
			func.Add (new BashCommand ("echo $" + EscapeString (release.UniqueName) + "_message"));
			func.Add (new BashCommand ("date +\"%F %T $" + EscapeString (release.UniqueName) + "_message\" >> " + Config.LOGDIR + "/upkg.log"));

			string statedir = root + (string) release.Settings.Variables["$STATEDIR"];
			
			// Call the build commands
			if (!release.Virtual)
			{
				string build_output_redirection = "";
				string output_redirection = "";
				if (Local.Verbose)
				{
					build_output_redirection = "2>&1 | tee " + statedir + "/logs/" + release.VersionedName + ".log";
				}
				else
				{
					build_output_redirection = "> " + statedir + "/logs/" + release.VersionedName + ".log 2>&1";
					output_redirection = "> /dev/null 2>&1";
				}

				func.Add (new BashCommand("mkdir -p " + release.Settings.Variables["$CACHEDIR"] + "/binaries"));

				// if this release uses sources from a local repository, don't use caching. Instead mount the repo dir into the cachedir
				if (!repo.Remote)
				{
					// check whether repo is readonly
					if (Local.Bootstrap)
						func.Add (new BashCommand ("repo_ro=$(repo_readonly " + repo.Url + ")"));
						
					func.Add (new SimpleCommand ("mount --bind " + repo.Url + " " + release.Settings.Variables["$CACHEDIR"], true).ToBashCommand(Local.Verbose));
				}

				func.Add (new SimpleCommand ("mkdir -p " + statedir + "/{packages,files,logs,scripts}", true).ToBashCommand (Local.Verbose));
				ArrayList env = new ArrayList ();
				env.Add ("HOME=/root");
				env.Add ("TERM=$TERM");
				env.Add ("UPKG_HOSTNAME=\"$(hostname -f)\"");
				env.Add (EscapeString (release.UniqueName) + "_mode=$" + EscapeString (release.UniqueName) + "_mode");

				// FIXME: don't set LD_PRELOAD when installing from binary to speedup installation
				if (release.UniqueName != "upkg-watch")
					env.Add ("LD_PRELOAD=/usr/lib/libupkg-watch.so");

				string env_string = String.Join (" ", (string[]) env.ToArray (typeof (string)));
				
				string scriptCall = filename;
				if (stream == null)
					scriptCall += "-";
				else
					scriptCall += " --build ";
				scriptCall += release.UniqueName;
				string command = "/usr/bin/env -i " + env_string + " /bin/bash --login +h " + scriptCall + " " + build_output_redirection;
				if (Local.Bootstrap && !release.Settings.Staging)
				{					
					// Make sure all neccessary directories are available in chroot
					func.Add (new SimpleCommand ("mkdir -p " + root + "/{dev,proc,sys}", true).ToBashCommand (Local.Verbose));
					func.Add (new SimpleCommand ("mkdir -p " + root + Local.CacheDir, true).ToBashCommand (Local.Verbose));
					func.Add (new SimpleCommand ("mount --rbind --make-rslave /dev " + root + "/dev", true).ToBashCommand (Local.Verbose));
					func.Add (new SimpleCommand ("mount --rbind --make-rslave /proc " + root + "/proc", true).ToBashCommand (Local.Verbose));
					func.Add (new SimpleCommand ("mount --rbind --make-rslave /sys " + root + "/sys", true).ToBashCommand (Local.Verbose));
					func.Add (new SimpleCommand ("mount --bind " + Local.CacheDir + " " + root + Local.CacheDir, true).ToBashCommand (Local.Verbose));
					if (!repo.Remote) {
						func.Add (new SimpleCommand ("mount --bind " + release.Settings.Variables["$CACHEDIR"] + " " + root + release.Settings.Variables["$CACHEDIR"], true).ToBashCommand (Local.Verbose));
					}
					
					// Call build-script in a clean chroot environment
					func.Add (new BashCommand ("unshare -n --loopback-setup chroot " + root + " " + command));
					// Check if our last command was successful
					// FIXME: we should not check pipestatus but normal return code if we're not using tee (i.e. in non-verbose mode)
					func.Add (new BashCommand ("[ ${PIPESTATUS[0]} -eq 0 ] || { echo '*** ERROR: " + release.VersionedName + "'; exit 1 ; }"));
					// Unmount
					// Delete log file if still there as the real one has been bzipped anyway
					func.Add (new BashCommand ("rm -f " + statedir + "/logs/" + release.VersionedName + ".log"));
				}
				else
				{
					// Call build-script in a clean environment
					func.Add (new BashCommand (command));
					// FIXME: we should not check pipestatus but normal return code if we're not using tee (i.e. in non-verbose mode)
					func.Add (new BashCommand ("[ ${PIPESTATUS[0]} -eq 0 ] || { echo '*** ERROR: " + release.VersionedName + "' ; exit 1 ; }"));
					// Delete log file if still there as the real one has been bzipped anyway
					func.Add (new BashCommand ("rm -f " + statedir + "/logs/" + release.VersionedName + ".log"));
				}
				
				// Copy state files if we're bootstrapping with a local repository
				if (Local.Bootstrap && !repo.Remote)
				{
					func.Add (UploadFunction (release));
					// only upload if repo is not readonly and is persistent storage (i.e. no tmpfs)
					func.Add (new SimpleCommand ("[ $repo_ro -eq 1 ] || upload " + output_redirection, false).ToBashCommand (Local.Verbose));
				}
				
				// unmount a possibly mounted places
				if (umount_str != String.Empty)
					func.Add (new SimpleCommand ("umount -l " + umount_str, false).ToBashCommand (Local.Verbose));
			}

			// Execute post-build script if we're building
			if (Local.Bootstrap && !release.Settings.Staging && release.PostBuild != null && release.PostBuild.Count > 0)
			{
				// move post-build commands to a separate function to be able to redirect output
				func.Add (PostBuildFunction (release));
				func.Add (new SimpleCommand ("postbuild " + (Local.Verbose ? "" : " > /dev/null 2>&1"), true).ToBashCommand (Local.Verbose));
			}
			
			// if we want to have this release in the package selection
			// write the appropriate .select file
			if (release.Selected || release.Settings.Keep)
			{
				string selectFileCommand = "echo \"Package: " + release.Package.Name;
				if (release.Tag != "default")
					selectFileCommand += "\nTag: " + release.Tag;
				selectFileCommand += "\"";
				func.Add (new BashCommand (selectFileCommand + " > " + statedir + "/packages/" + release.UniqueName + ".select"));
			}
			
			// remove trap
			func.Add (new BashCommand ("trap \"\" INT TERM EXIT"));

			return (func);
		}

		protected BashFunction PostBuildFunction (ReleaseSpecification release)
		{
			BashFunction func = new BashFunction ("postbuild");

			foreach (Command cmd in release.PostBuild)
				func.Add (cmd.ToBashCommand (true));

			func.Add (new BashCommand ("true"));	// the return code of the last command defines the return code of the whole function
			
			return (func);
		}

		protected BashFunction DownloadFunction (ReleaseSpecification release)
		{
			BashFunction func = new BashFunction ("download");

			string repo = release.Package.Specification.Repository.Url;

			if (Local.AllowBinaryInstall && release.Settings.AllowBinaryInstall)
			{
				func.Add (new DownloadCommand (release.BinaryName, repo, (string) release.Settings.Variables["$CACHEDIR"], "binaries", false).ToBashCommand (true));
			}
			
			// Fetch the sources if this package needs any
			// don't even check whether we need the sources unless source build is enabled or
			// we're in --disable-binary mode or the package doesn't support binary installation
			if (release.BuildSources.Count > 0 &&
				(Local.AllowSourceBuild || !release.Settings.AllowBinaryInstall))
			{
				BashList sourceDownloadList = new BashList ();
			
				// only fetch sources fi we're building the package from source
				if (Local.AllowBinaryInstall && release.Settings.AllowBinaryInstall)
				{
					func.Add (new BashCommand ("if [ ! -e " + release.Settings.Variables["$CACHEDIR"] + "/binaries/" + release.BinaryName + " ]"));
					func.Add (new BashCommand ("then"));
					
					sourceDownloadList.Indent = true;
				}
					
				sourceDownloadList.Add (new BashCommand ("mkdir -p " + release.Settings.Variables["$CACHEDIR"] + "/sources/" + release.Package.Name));
				foreach (ResolvableString file in release.BuildSources)
				{
					sourceDownloadList.Add (new DownloadCommand (file, repo, (string) release.Settings.Variables["$CACHEDIR"], "sources/" + release.Package.Name, true).ToBashCommand (true));
				}

				func.Add (sourceDownloadList);

				if (Local.AllowBinaryInstall && release.Settings.AllowBinaryInstall)
					func.Add (new BashCommand ("fi"));
			}
			else
			{
				// ensure that download doesn't fail just because the package doesn't need any sources
				func.Add (new BashCommand ("true"));
			}
			
			return (func);
		}

		protected BashFunction UploadFunction (ReleaseSpecification release)
		{
			BashFunction func = new BashFunction ("upload");

			string root = "";
			if (Local.Bootstrap && !release.Settings.Staging)
				root = release.Settings.Variables["$CHROOTDIR"] as string;
			string statedir = root + (string) release.Settings.Variables["$STATEDIR"];

			string repo = release.Package.Specification.Repository.Url;

			BashList uploadList = new BashList ();

			if (Local.AllowSourceBuild && release.Settings.AllowBinaryInstall)
			{
				// Copy state files only if we've just built a binary (or this package doesn't support building binaries at all)
				string binary = release.Settings.Variables["$CACHEDIR"] + "/binaries/" + release.BinaryName;
				func.Add (new BashCommand ("if [ \"$" + EscapeString (release.UniqueName) + "_mode\" = \"build\" -a -e " + binary + " ] ; then"));
				
				uploadList.Indent = true;
			}
		
			// Copy all sort of information into the repository
			// do this for <binary>no</binary> packages, too
			// that's the reason why it's out of the if [ "$BUILDING" = "1" ] block
			uploadList.Add (new SimpleCommand ("mkdir -p " + repo + "/state/{info,files,logs}", false).ToBashCommand (Local.Verbose));
			uploadList.Add (new SimpleCommand ("install " + statedir + "/packages/" + release.VersionedName + ".info " + repo + "/state/info", false).ToBashCommand (true));
			uploadList.Add (new SimpleCommand ("install " + statedir + "/files/" + release.VersionedName + ".* " + repo + "/state/files", false).ToBashCommand (true));
			uploadList.Add (new SimpleCommand ("install " + statedir + "/logs/" + release.VersionedName + ".* " + repo + "/state/logs", false).ToBashCommand (true));
			
			func.Add (uploadList);
		
			if (Local.AllowSourceBuild && release.Settings.AllowBinaryInstall)
			{
				func.Add (new BashCommand ("fi"));
			}

			return (func);
		}

		// writes the build function of a package
		protected BashFunction ReleaseBuildFunction (ReleaseSpecification release)
		{
			// Check for non valid parameters
			if (release == null)
				throw new ArgumentNullException("release");
		
			// Don't add package if we're installing and package is not installable
			if (!Local.Bootstrap && release.Settings.Staging)
				return (null);
				
			BashFunction func = new BashFunction ("build_" + release.UniqueName);

			// Prepare removing the old package if installed (using upkg-remove), do not do this for staging packages
			if (!release.Settings.Staging)
			{
				func.Add (new BashCommand ("[ -e " + release.Settings.Variables["$STATEDIR"] + "/packages/" + release.UniqueName + " ] && upkg-remove --pre-upgrade " + release.UniqueName + " 2>/dev/null"));
			}
			
			// Clean the environment
			func.Add (new SimpleCommand ("unset CFLAGS CXXFLAGS CPPFLAGS DEFS LDFLAGS LD_LIBRARY_PATH LD_PRELOAD LIBS LANG LANGUAGE LC_ALL LC_CTYPE LC_NUMERIC LC_TIME LC_COLLATE LC_MONETARY LC_PAPER LC_NAME LC_ADDRESS LC_TELEPHONE LC_MEASUREMENT LC_IDENTIFICATION PATH", true).ToBashCommand (true));
			// Set the environment
			foreach (string env in release.Settings.Environment.Keys)
			{
				func.Add (new SimpleCommand ("export " + env + "=\"" + release.Settings.Environment[env] + "\"", true).ToBashCommand (true));
			}
			// Print the eviroment
			func.Add (new SimpleCommand ("env | sort", false).ToBashCommand (true));

			// Execute pre-install script
			if (release.PreInst != null)
			{
				foreach (Command cmd in release.PreInst)
					func.Add (cmd.ToBashCommand (true));
			}
			
			// command list for source build
			BashList buildList = new BashList ();

			if (Local.AllowBinaryInstall && release.Settings.AllowBinaryInstall)
			{
				// Check whether we're installing a binary
				func.Add (new BashCommand ("if [ -e " + release.Settings.Variables["$CACHEDIR"] + "/binaries/" + release.BinaryName + " ]"));
				func.Add (new BashCommand ("then"));
				
				BashList installList = new BashList ();
				installList.Indent = true;

				// Extract the binary package
				// -P is necessary for correctly extracting binary packages with potential dangerous symlinks (for example Xorg)
				installList.Add (new SimpleCommand ("tar -Pxf " + release.Settings.Variables["$CACHEDIR"] + "/binaries/" + release.BinaryName + " -C /", true).ToBashCommand (true));
				
				func.Add (installList);

				func.Add (new BashCommand ("else"));

				buildList.Indent = true;
			}

			func.Add (buildList);

			// There is no binary package ==> build it
			
			if (release.Settings.Register)
			{
				// Setup temporary files
				buildList.Add (new BashCommand ("mkdir -p /tmp"));
				buildList.Add (new BashCommand ("date +%s > /tmp/upkg.stamp"));
				buildList.Add (new BashCommand ("rm -f /tmp/upkg-watch{,.dirs}.log"));
			}
			else
			{
				// if the package is not registred no digest must be build
				buildList.Add (new BashCommand ("unset LD_PRELOAD"));
			}

			// Execute all package build commands
			foreach (Command cmd in release.BuildScript)
			{
				buildList.Add (cmd.ToBashCommand (true));
			}

			// The package has been built, now create / process package info files (info, digest, dirs, links, config, log)
			if (release.Settings.Register)
			{
				buildList.Add (new BashCommand ("echo 0 > /tmp/upkg.size"));
				buildList.Add (new BashCommand ("unset LD_PRELOAD"));

				if (release.BuildAddFiles != null)
				{
					// add files which miss in the watch logs
					foreach (ResolvableString file in release.BuildAddFiles)
					{
						buildList.Add (new BashCommand ("find " + file + " ! -type d >> /tmp/upkg-watch.log"));
						buildList.Add (new BashCommand ("find " + file + " -type d >> /tmp/upkg-watch.dirs.log"));
					}
				}
				
				BashList watchLoop = new BashList ();
				watchLoop.Indent = true;
				BashList watchCase = new BashList ();
				watchCase.Indent = true;

				// process upkg-watch.log: ignore files that should be ignored and strip binaries
				buildList.Add (new BashCommand ("echo Computing checksums for all installed files..."));
				buildList.Add (new BashCommand ("[ -e /tmp/upkg-watch.log ] && cat /tmp/upkg-watch.log | sort -u | ("));

				watchLoop.Add (new BashCommand ("while read file; do"));

				watchCase.Add (new BashCommand ("case \"$file\" in"));
				foreach (ResolvableString file in release.BuildIgnoreFiles)
					watchCase.Add (new BashCommand (file + ") continue ;;"));
				watchCase.Add (new BashCommand ("esac"));
				watchCase.Add (new BashCommand ("[ -L \"$file\" ] && echo $(readlink \"$file\") $file >> " + release.Settings.Variables["$STATEDIR"] + "/files/" + release.VersionedName + ".links"));
				watchCase.Add (new BashCommand ("[ -f \"$file\" -a ! -L \"$file\" ] || continue")); // log only files which still exist; symlinks shouldn't be stripped
				if (release.BuildStrip)
				{
					foreach (Command cmd in release.Strip)
						watchCase.Add (cmd.ToBashCommand (false));
				}
				watchCase.Add (new BashCommand ("echo $(($(cat /tmp/upkg.size) + $(du -b \"$file\" | sed 's/\\t.*//'))) > /tmp/upkg.size"));
				watchCase.Add (new BashCommand ("md5sum --text \"$file\" >> " + release.Settings.Variables["$STATEDIR"] + "/files/" + release.VersionedName + ".digest"));
				watchLoop.Add (watchCase);
				watchLoop.Add (new BashCommand ("done"));

				buildList.Add (watchLoop);
				buildList.Add (new BashCommand (")"));
				
				watchLoop = new BashList ();
				watchLoop.Indent = true;
				watchCase = new BashList ();
				watchCase.Indent = true;

				// process upkg-watch.dirs.log
				buildList.Add (new BashCommand ("[ -e /tmp/upkg-watch.dirs.log ] && cat /tmp/upkg-watch.dirs.log | sort -u | ("));

				watchLoop.Add (new BashCommand ("while read dir; do"));

				watchCase.Add (new BashCommand ("[ -d \"$dir\" ] || continue"));
				watchCase.Add (new BashCommand ("case \"$dir\" in"));
				foreach (ResolvableString file in release.BuildIgnoreFiles)
					watchCase.Add (new BashCommand (file + ") continue ;;"));
				watchCase.Add (new BashCommand ("esac"));
				watchCase.Add (new BashCommand ("echo \"$dir\" >> " + release.Settings.Variables["$STATEDIR"] + "/files/" + release.VersionedName + ".dirs"));

				watchLoop.Add (watchCase);
				watchLoop.Add (new BashCommand ("done"));

				buildList.Add (watchLoop);
				buildList.Add (new BashCommand (")"));

				// create $VERSIONEDNAME.config file
				foreach (ConfigFile configFile in release.ConfigFiles)
				{
					buildList.Add (new BashCommand ("echo " + configFile.Default + " " + configFile.Dest + " >> " + release.Settings.Variables["$STATEDIR"] + "/files/" + release.VersionedName + ".config"));
				}
				
				if (release.PreRm != null && release.PreRm.Count > 0)
				{
					// create a prerm script
					buildList.Add (new BashCommand ("cat > " + release.Settings.Variables["$STATEDIR"] + "/scripts/" + release.VersionedName + ".prerm << \"EOF\""));
					BashList preRmList = new BashList ();
					preRmList.Preformatted = true;

					preRmList.Add (new BashCommand ("#!/bin/bash"));
					foreach (Command cmd in release.PreRm)
						preRmList.Add (cmd.ToBashCommand (true));
					preRmList.Add (new BashCommand ("EOF"));
					
					buildList.Add (preRmList);
				}

				// mark package as installed
				buildList.Add (new BashCommand ("touch " + release.Settings.Variables["$STATEDIR"] + "/packages/" + release.VersionedName));
				buildList.Add (new BashCommand ("ln -sf " + release.VersionedName + " " + release.Settings.Variables["$STATEDIR"] + "/packages/" + release.UniqueName));
			}

			string compressCmd, catCmd, tarflag;
			if (Global.Compression == "gz")
			{
				compressCmd = "gzip";
				catCmd = "zcat";
				tarflag = "-z";
			}
			else if (Global.Compression == "bz2")
			{
				compressCmd = "bzip2";
				catCmd = "bzcat";
				tarflag = "-j";
			}
			else if (Global.Compression == "xz")
			{
				compressCmd = "xz";
				catCmd = "xzcat";
				tarflag = "-J";
			}
			else if (Global.Compression == "zst")
			{
				compressCmd = "zstd --rm";
				catCmd = "zstdcat";
				tarflag = "--zstd";
			}
			else
			{
				throw new NotImplementedException ("Unsupported compression format: " + Global.Compression);
			}
			
			// Compress digest and log
			if (release.Settings.Register)
				buildList.Add (new SimpleCommand (compressCmd + " -f " + release.Settings.Variables["$STATEDIR"] + "/{files,logs}/" + release.VersionedName + ".*", false).ToBashCommand (true));
			else
				buildList.Add (new SimpleCommand (compressCmd + " -f " + release.Settings.Variables["$STATEDIR"] + "/logs/" + release.VersionedName + ".*", false).ToBashCommand (true));

			// create the .info file
			if (release.Settings.Register)
			{
				string infoFileCommand = "echo \"Package: " + release.Package.Name + "\n";
				if (release.Tag != "default")
					infoFileCommand += "Tag: " + release.Tag + "\n";
				infoFileCommand += "Version: " + release.Version + "-" + release.Revision + "\n";
				infoFileCommand += "Branch: " + release.BinaryBranch + "\n";
				infoFileCommand += "Installed-Size: $((($(cat /tmp/upkg.size) + 512) / 1024)) KB\n";
				infoFileCommand += "Build-Duration: $((($(date +%s) - $(cat /tmp/upkg.stamp) + 30) / 60)) min\n";
				infoFileCommand += "Build-Date: $(date -u \"+%F %R:%S %z\")\n";
				infoFileCommand += "Build-Host: $UPKG_HOSTNAME\"";
				buildList.Add (new BashCommand (infoFileCommand + " > " + release.Settings.Variables["$STATEDIR"] + "/packages/" + release.VersionedName + ".info"));

				// create a binary tarball unless --source-only has been specified or it has been explicitly deactived for the package or we're installing in a running system or actual global branch is not equal to release branch
				if (Local.AllowBinaryInstall && release.Settings.AllowBinaryInstall && Local.Bootstrap && Local.Branch == release.BinaryBranch)
				{
					// don't create the binary tarball if we're upgrading as this could lead to a non-clean tarball
					buildList.Add (new BashCommand ("if [ ! -e " + release.Settings.Variables["$STATEDIR"] + "/packages/" + release.UniqueName + ".saved ] ; then"));
					
					BashList tarballList = new BashList ();
					tarballList.Indent = true;

					// create binary tarball
					tarballList.Add (new BashCommand ("echo Creating binary tarball " + release.BinaryName));
					tarballList.Add (new BashCommand (catCmd + " " + release.Settings.Variables["$STATEDIR"] + "/files/" + release.VersionedName + ".digest." + Global.Compression + " 2>/dev/null | ("));
					
					BashList innerTarballList = new BashList ();
					innerTarballList.Indent = true;
					
					// add dirs to the tarball
					innerTarballList.Add (new BashCommand (catCmd + " " + release.Settings.Variables["$STATEDIR"] + "/files/" + release.VersionedName + ".dirs." + Global.Compression + " 2>/dev/null | ("));
					innerTarballList.Add (new BashCommand ("\twhile read dir ; do"));
					innerTarballList.Add (new BashCommand ("\t\techo /.$dir"));
					innerTarballList.Add (new BashCommand ("\tdone )"));

					// add links to the tarball
					innerTarballList.Add (new BashCommand (catCmd + " " + release.Settings.Variables["$STATEDIR"] + "/files/" + release.VersionedName + ".links." + Global.Compression + " 2>/dev/null | ("));
					innerTarballList.Add (new BashCommand ("\twhile read target link ; do"));
					innerTarballList.Add (new BashCommand ("\t\techo /.$link"));
					innerTarballList.Add (new BashCommand ("\tdone )"));

					innerTarballList.Add (new BashCommand ("while read md5sum file ; do"));
					innerTarballList.Add (new BashCommand ("\techo /.$file"));
					innerTarballList.Add (new BashCommand ("done"));

					// add the files to the tarball which are not part of the digest
					innerTarballList.Add (new BashCommand ("echo /." + release.Settings.Variables["$STATEDIR"] + "/packages/" + release.UniqueName));
					innerTarballList.Add (new BashCommand ("echo /." + release.Settings.Variables["$STATEDIR"] + "/packages/" + release.VersionedName));
					innerTarballList.Add (new BashCommand ("echo /." + release.Settings.Variables["$STATEDIR"] + "/files/" + release.VersionedName + ".digest." + Global.Compression));
					innerTarballList.Add (new BashCommand ("echo /." + release.Settings.Variables["$STATEDIR"] + "/files/" + release.VersionedName + ".dirs." + Global.Compression));
					innerTarballList.Add (new BashCommand ("echo /." + release.Settings.Variables["$STATEDIR"] + "/files/" + release.VersionedName + ".links." + Global.Compression));
					innerTarballList.Add (new BashCommand ("echo /." + release.Settings.Variables["$STATEDIR"] + "/files/" + release.VersionedName + ".config." + Global.Compression));
					innerTarballList.Add (new BashCommand ("echo /." + release.Settings.Variables["$STATEDIR"] + "/logs/" + release.VersionedName + ".log." + Global.Compression));
					innerTarballList.Add (new BashCommand ("echo /." + release.Settings.Variables["$STATEDIR"] + "/packages/" + release.VersionedName + ".info"));
					if (release.PreRm != null && release.PreRm.Count > 0)
						innerTarballList.Add (new BashCommand ("echo /." + release.Settings.Variables["$STATEDIR"] + "/scripts/" + release.VersionedName + ".prerm"));
						
					tarballList.Add (innerTarballList);

					// Create tarball first with temporary file name to avoid downloading partially created binaries
					tarballList.Add (new BashCommand (") | tar " + tarflag + " -cf " + release.Settings.Variables["$CACHEDIR"] + "/binaries/." + release.BinaryName + ".upkg --no-recursion -T - 2>/dev/null"));
					
					// Rename temporary file to final binary name
					tarballList.Add (new BashCommand ("mv " + release.Settings.Variables["$CACHEDIR"] + "/binaries/." + release.BinaryName + ".upkg " + release.Settings.Variables["$CACHEDIR"] + "/binaries/" + release.BinaryName));
					
					buildList.Add (tarballList);
					
					buildList.Add (new BashCommand ("fi"));
				}

				if (Local.AllowBinaryInstall && release.Settings.AllowBinaryInstall)
					func.Add (new BashCommand ("fi"));
				
				// Remove the old package if installed (using upkg-remove), do not do this for staging packages
				if (!release.Settings.Staging)
				{
					func.Add (new BashCommand ("[ -e " + release.Settings.Variables["$STATEDIR"] + "/packages/" + release.UniqueName + ".saved ] && upkg-remove --post-upgrade " + release.UniqueName + " 2>/dev/null"));
				}
			}

			if (release.PostInst != null)
			{
				// Execute merged post-inst scripts (first global, then tag, then package script)
				foreach (Command cmd in release.PostInst)
					func.Add (cmd.ToBashCommand (true));
			}
			
			// clean up temporary files
			if (release.Settings.Register)
				func.Add (new BashCommand ("rm -f /tmp/upkg-watch{,.dirs}.log /tmp/upkg.{stamp,size}"));
			
			return (func);
		}
		
		protected BashFunction FinalizeFunction ()
		{
			BashFunction func = new BashFunction ("finalize");

			foreach (Command cmd in ((BranchSpecification) Global.Branches.Current).Finalize)
				func.Add (cmd.ToBashCommand (true));
			
			func.Add (new BashCommand ("true"));	// we need at least one command in the function
								// and the return code of the last command defines the return code of the whole function
						
			return (func);
		}
		
		protected BashFunction RepoReadonlyFunction ()
		{
			BashFunction func = new BashFunction ("repo_readonly");
			
			func.Add (new BashCommand ("fstype=$(find $1 -maxdepth 0 -printf '%F')"));
			func.Add (new BashCommand ("if [ \"$fstype\" = \"tmpfs\" ] || [ \"$fstype\" = \"ramfs\" ] || [ \"$fstype\" = \"iso9660\" ] || [ \"$fstype\" = \"squashfs\" ] || [ \"$fstype\" = \"unionfs\" ]"));
			func.Add (new BashCommand ("then"));
			func.Add (new BashCommand ("\techo 1"));
			func.Add (new BashCommand ("elif touch -r $1 $1"));
			func.Add (new BashCommand ("then"));
			func.Add (new BashCommand ("\techo 0"));
			func.Add (new BashCommand ("else"));
			func.Add (new BashCommand ("\techo 1"));
			func.Add (new BashCommand ("fi"));
			
			return (func);
		}
		
		protected BashCommand SimpleChrootCommand (string command, bool check, bool verbose)
		{
			BashList list = new BashList ();

			string root = (string) ((BranchSpecification) Global.Branches.Current).Settings.Variables["$CHROOTDIR"];

			// Make sure all neccessary directories are available in chroot
			list.Add (new SimpleCommand ("mkdir -p " + root + "/{dev,proc,sys}", true).ToBashCommand (verbose));
			list.Add (new SimpleCommand ("mkdir -p " + root + Local.CacheDir, true).ToBashCommand (verbose));
			list.Add (new SimpleCommand ("mount --rbind --make-rslave /dev " + root + "/dev", true).ToBashCommand (verbose));
			list.Add (new SimpleCommand ("mount --rbind --make-rslave /proc " + root + "/proc", true).ToBashCommand (verbose));
			list.Add (new SimpleCommand ("mount --rbind --make-rslave /sys " + root + "/sys", true).ToBashCommand (verbose));
			list.Add (new SimpleCommand ("mount --bind " + Local.CacheDir + " " + root + Local.CacheDir, true).ToBashCommand (verbose));
			// Call build-script in a clean chroot environment
			list.Add (new BashCommand ("unshare -n --loopback-setup chroot " + root + " " + command));
			list.Add (new BashCommand ("ret=$?"));
			// Unmount
			list.Add (new SimpleCommand ("umount -l " + root + "{/dev,/proc,/sys," + Local.CacheDir + "}", false).ToBashCommand (verbose));
			// don't fail if /upkg/bin/bash doesn't exist, that just means that e.g. a cd/dvd has been built
			if (check)
				list.Add (new SimpleCommand ("[ $ret -eq 0 ] || [ ! -e /upkg/bin/bash ]", true).ToBashCommand (verbose));
			
			return (list);
		}
		
		public BashCommand MainFunction ()
		{
			BashList argCheck = new BashList ();
		
			argCheck.Add (new BashCommand ("if [ \"$1\" = \"--finalize\" ]"));
			argCheck.Add (new BashCommand ("then"));
			argCheck.Add (new BashCommand ("\tfinalize"));

			argCheck.Add (new BashCommand("elif [ $# -le 1 ]\nthen"));

			BashList main = new BashList ();
			main.Indent = true;

			// Set the packages variable to all package names
			main.Add (new BashCommand ("RELEASES='" + String.Join (" ", (string[]) releaseList.ToArray (typeof (string))) + "'"));
			main.Add (new BashCommand ("CHECK_RELEASES='" + String.Join (" ", (string[]) checkReleaseList.ToArray (typeof (string))) + "'"));
			
			// Check all releases
			main.Add (new BashCommand ("for release in $CHECK_RELEASES"));
			main.Add (new BashCommand ("do"));
			main.Add (new BashCommand ("\tcheck_$release"));
			main.Add (new BashCommand ("done"));
			
			main.Add (new BashCommand ("if [ \"$SKIP_RELEASES\" ]"));
			main.Add (new BashCommand ("then"));
			main.Add (new BashCommand ("\techo -e \"\\033[1mThe following packages have been skipped as no binary is available yet:\\033[22m\""));
			main.Add (new BashCommand ("\techo $SKIP_RELEASES"));
			main.Add (new BashCommand ("\techo"));
			main.Add (new BashCommand ("fi"));
			
			if (!Local.IgnoreSelection)
			{
				// Remove all packages not referenced
				string path;
				Settings settings = ((BranchSpecification)Global.Branches.Current).Settings;
				
				if (Local.Bootstrap)
					path = Path.Combine ((string)settings.Variables["$CHROOTDIR"] + (string)settings.Variables["$STATEDIR"], "packages");
				else
					path = Path.Combine ((string)settings.Variables["$STATEDIR"], "packages");
				
				main.Add (new BashCommand ("if [ -d " + path + " ]"));
				main.Add (new BashCommand ("then"));
				
				BashList findReleases = new BashList ();
				findReleases.Indent = true;
				findReleases.Add (new BashCommand ("releasesfile=\"$(mktemp)\""));
				findReleases.Add (new BashCommand ("installedfile=\"$(mktemp)\""));
				findReleases.Add (new BashCommand ("echo \"$RELEASES\" | tr ' ' '\\n' | sort > $releasesfile"));
				findReleases.Add (new BashCommand ("pushd " + path + " > /dev/null"));
				findReleases.Add (new BashCommand ("find . -type l | cut -d '/' -f 2 | sort > $installedfile"));
				findReleases.Add (new BashCommand ("popd > /dev/null"));
				findReleases.Add (new BashCommand ("for rel in $(diff $installedfile $releasesfile | grep '^<' | cut -b 3-)"));
				findReleases.Add (new BashCommand ("do"));
				findReleases.Add (new BashCommand ("\tREMOVE_RELEASES=\"$REMOVE_RELEASES $rel\""));
				findReleases.Add (new BashCommand ("done"));
				findReleases.Add (new BashCommand ("rm -f $releasesfile $installedfile"));
				main.Add (findReleases);

				main.Add (new BashCommand ("fi"));
				
				// Print actions to be taken
				main.Add (new BashCommand ("if [ \"$REMOVE_RELEASES\" ]"));
				main.Add (new BashCommand ("then"));
				main.Add (new BashCommand ("\techo -e \"\\033[1mThe following packages will be removed:\\033[22m\""));
				main.Add (new BashCommand ("\techo $REMOVE_RELEASES"));
				main.Add (new BashCommand ("\techo"));
				main.Add (new BashCommand ("fi"));
			}
			
			main.Add (new BashCommand ("if [ \"$EXTRA_RELEASES\" ]"));
			main.Add (new BashCommand ("then"));
			main.Add (new BashCommand ("\techo -e \"\\033[1mThe following extra packages will be installed:\\033[22m\""));
			main.Add (new BashCommand ("\techo $EXTRA_RELEASES"));
			main.Add (new BashCommand ("\techo"));
			main.Add (new BashCommand ("fi"));
			
			main.Add (new BashCommand ("if [ \"$UPGRADE_RELEASES\" ]"));
			main.Add (new BashCommand ("then"));
			main.Add (new BashCommand ("\techo -e \"\\033[1mThe following packages will be upgraded:\\033[22m\""));
			main.Add (new BashCommand ("\techo $UPGRADE_RELEASES"));
			main.Add (new BashCommand ("\techo"));
			main.Add (new BashCommand ("fi"));
			
			main.Add (new BashCommand ("if [ \"$SOURCE_RELEASES\" ]"));
			main.Add (new BashCommand ("then"));
			main.Add (new BashCommand ("\techo -e \"\\033[1mThe following packages will be built from source:\\033[22m\""));
			main.Add (new BashCommand ("\techo $SOURCE_RELEASES"));
			main.Add (new BashCommand ("\techo"));
			main.Add (new BashCommand ("fi"));
			
			main.Add (new BashCommand ("if [ -z \"$EXTRA_RELEASES\" ] && [ -z \"$UPGRADE_RELEASES\" ] && [ -z \"$REMOVE_RELEASES\" ]"));
			main.Add (new BashCommand ("then"));
			main.Add (new BashCommand ("\techo System already up-to-date."));
			main.Add (new BashCommand ("\texit 0"));
			main.Add (new BashCommand ("fi"));

			main.Add (new BashCommand ("if [ \"$UPDATE_RELEASES\" ]"));
			main.Add (new BashCommand ("then"));
			main.Add (new BashCommand ("\techo -e \"\\033[1mThe following packages will be updated:\\033[22m\""));
			main.Add (new BashCommand ("\techo $UPDATE_RELEASES"));
			main.Add (new BashCommand ("\techo"));
			main.Add (new BashCommand ("fi"));
			
			// Ask user to confirm as long as --force hasn't been specified
			main.Add (new BashCommand ("if [ \"$1\" != \"--force\" ]"));
			main.Add (new BashCommand ("then"));
			
			BashList confirm = new BashList ();
			confirm.Indent = true;
			confirm.Add (new BashCommand ("echo -n \"Do you want to continue? [Y/n] \""));
			confirm.Add (new BashCommand ("read yes"));
			confirm.Add (new BashCommand ("if [ \"$yes\" != \"\" -a \"$yes\" != \"y\" -a \"$yes\" != \"Y\" -a \"$yes\" != \"yes\" ]"));
			confirm.Add (new BashCommand ("then"));
			confirm.Add (new BashCommand ("\techo Abort."));
			confirm.Add (new BashCommand ("\texit 1"));
			confirm.Add (new BashCommand ("fi"));
			main.Add (confirm);

			main.Add (new BashCommand ("fi"));
			
			if (Local.Bootstrap && Local.CacheDirChroot)
			{
				// installation from cd
				string root = (string) ((BranchSpecification) Global.Branches.Current).Settings.Variables["$CHROOTDIR"];
				
				main.Add (new BashCommand ("mkdir -p " + root + Local.CacheDir));
				main.Add (new BashCommand ("cp " + filename + "* " + root + Local.CacheDir + " 2>/dev/null"));
				main.Add (new BashCommand ("grep -q " + Local.CacheDir + " /proc/mounts || mount --bind " + root + Local.CacheDir + " " + Local.CacheDir + " 2>/dev/null"));
			}

			// Download all required files
			main.Add (new BashCommand ("for release in $INSTALL_RELEASES"));
			main.Add (new BashCommand ("do"));
			main.Add (new BashCommand ("\tdownload_$release"));
			main.Add (new BashCommand ("done"));

			// Remove all non referenced packages
			main.Add (new BashCommand ("for release in $REMOVE_RELEASES"));
			main.Add (new BashCommand ("do"));
			main.Add (new SimpleCommand ("\tupkg-remove --force $release" + (Local.Verbose ? "" : " 2>/dev/null"), true).ToBashCommand (Local.Verbose));
			main.Add (new BashCommand ("done"));
			
			// Install all releases requiring an upgrade
			main.Add (new BashCommand ("for release in $INSTALL_RELEASES"));
			main.Add (new BashCommand ("do"));
			main.Add (new BashCommand ("\tinstall_$release"));
			main.Add (new BashCommand ("done"));
			
			if (Local.Bootstrap)
				main.Add (SimpleChrootCommand (filename + " --finalize" + (Local.Verbose ? "" : " > /dev/null 2>&1"), true, Local.Verbose));
			else
				main.Add (new SimpleCommand (filename + " --finalize" + (Local.Verbose ? "" : " > /dev/null 2>&1"), true).ToBashCommand (Local.Verbose));
			
			if (Local.Bootstrap && Local.CacheDirChroot)
			{
				main.Add (new BashCommand ("umount " + Local.CacheDir + " 2>/dev/null"));
			}
			
			argCheck.Add (main);

			// if script gets called as SCRIPT --build UNIQUENAME
			// just build the specified release
			argCheck.Add (new BashCommand ("elif [ \"$1\" = \"--build\" ]"));
			argCheck.Add (new BashCommand ("then"));
			argCheck.Add (new BashCommand ("\tbuild_$2"));
			argCheck.Add (new BashCommand ("fi"));
			
			return (argCheck);
		}
		
		// Translate characters not valid in a bash variable
		protected static string EscapeString (string s)
		{
			return (s.Replace ("-", "_").Replace ("+", "_").Replace (".", "_"));
		}
		
		private BashFunction AddReleaseFunction ()
		{
			BashFunction function = new BashFunction ("add_release");
			
			function.Add (new BashCommand ("INSTALL_RELEASES=\"$INSTALL_RELEASES $3\""));
			function.Add (new BashCommand ("[ -e $1/packages/$3 ] && UPGRADE_RELEASES=\"$UPGRADE_RELEASES $3\" || EXTRA_RELEASES=\"$EXTRA_RELEASES $3\""));
			function.Add (new BashCommand ("[ \"$2\" = \"source\" ] && SOURCE_RELEASES=\"$SOURCE_RELEASES $3\""));
			
			return (function);
		}
	}
}
