// DownloadCommand.cs
//
// Cares about downloads of sources and binaries
//
// Copyright (C) 2005 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.Collections;
using System.IO;
using System.Xml;

namespace Upkg
{
	class DownloadCommand : Command
	{
		public readonly string File;
		public readonly string Source;
		public readonly string Destination;
		public readonly string Path;
		public readonly bool Check;
		
		// constructor
		public DownloadCommand (string file, string source, string destination, string path, bool check)
		{
			if (file == null)
				throw new ArgumentNullException ("file");
			if (source == null)
				throw new ArgumentNullException ("source");
			if (destination == null)
				throw new ArgumentNullException ("destination");
			if (path == null)
				throw new ArgumentNullException ("path");
				
			File = file;
			Source = source;
			Destination = destination;
			Path = path;
			Check = check;
		}
		
		// the shell func
		public override BashCommand ToBashCommand (bool verbose)
		{
			BashList list = new BashList ();
			
			string cond = "-e " + Destination + "/" + Path + "/" + System.IO.Path.GetFileName (File);

			bool remoteFile = File.StartsWith ("http://") || File.StartsWith ("https://");

			// if we're using a local repo, try to copy the source from there
			if (!Source.StartsWith ("http://") && !Source.StartsWith ("https://"))
			{
				list.Add (new SimpleCommand ("[ " + cond + " ] || cp -f " + Source + "/" + Path + "/" + System.IO.Path.GetFileName (File) + " " + Destination + "/" + Path, Check && !remoteFile).ToBashCommand (verbose));
			}
			// if we're using a remote repo, download the source from there only if we don't have an other source url
			else if (!remoteFile)
			{
				list.Add (new SimpleCommand ("[ " + cond + " ] || wget -P " + Destination + "/" + Path + " " + Source + "/" + Path + "/" + File, Check).ToBashCommand (verbose));
			}

			if (remoteFile)
			{
				list.Add (new SimpleCommand ("[ " + cond + " ] || wget -P " + Destination + "/" + Path + " " + File, Check).ToBashCommand (verbose));
			}
				
			return (list);
		}
	}
}
