// Graph.cs
//
// Represents a directed graph
//
// Copyright (C) 2004 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>


using System;
using System.Collections;
using System.Diagnostics;

public class Graph
{
	struct Relation
	{
		public Relation(object predecessor, object successor)
		{
			Predecessor = predecessor;
			Successor = successor;
		}
		public object Predecessor;
		public object Successor;
	}
	
	public void Add(object predecessor, object successor)
	{
		if (predecessor == null)
			throw (new ArgumentNullException ("predecessor"));
		if (successor == null)
			throw (new ArgumentNullException ("successor"));

		relations.Add(new Relation(predecessor, successor));
	}
	
	public ArrayList TopologicalSort()
	{
		ArrayList result = new ArrayList();
		int itemCount;
	
		items = new Hashtable();
		foreach (Relation relation in relations)
		{
			Item predecessor;
			Item successor;

			if (items.ContainsKey(relation.Predecessor))
				predecessor = (Item) items[relation.Predecessor];
			else
			{
				predecessor = new Item(relation.Predecessor);
				items.Add(relation.Predecessor, predecessor);
			}
			if (items.ContainsKey(relation.Successor))
				successor = (Item) items[relation.Successor];
			else
			{
				successor = new Item(relation.Successor);
				items.Add(relation.Successor, successor);
			}
			
			if (predecessor != successor)
			{
				successor.Count++;
				predecessor.Successors.Add(successor);
			}
		}
		itemCount = items.Count;

		itemQueue = new Queue();
		// search for zero items (items without a predecessor)
		foreach (Item item in items.Values)
		{
			if (item.Count == 0)
			{
				itemQueue.Enqueue(item);
			}
		}
		while (itemQueue.Count > 0)
		{
			Item item = (Item) itemQueue.Dequeue();
			result.Add(item.Obj);
			itemCount--;
			foreach (Item successor in item.Successors)
			{
				successor.Count--;
				Debug.Assert(successor.Count >= 0);
				if (successor.Count == 0)
					itemQueue.Enqueue(successor);
			}
		}
		Debug.Assert(itemCount >= 0);
		if (itemCount > 0)
		{
			// graph contains a loop
			throw new Exception("Graph has a loop");
		}
		return result;
	}

	protected ArrayList relations = new ArrayList();

	class Item
	{
		public Item(object node)
		{
			Obj = node;
			Count = 0;
			Successors = new ArrayList();
		}
		public Object Obj;
		public int Count;
		public ArrayList Successors;
	}
	
	protected Hashtable items;
	protected Queue itemQueue;
}
