// InstallCommand.cs
//
// Holds all information a install command element has
//
// Copyright (C) 2004 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>


using System;
using System.Collections;
using System.IO;
using System.Xml;

namespace Upkg
{
	class InstallCommand : Command
	{
		public readonly ResolvableString Name;
		public readonly ResolvableString Dest;
		public readonly string Mode;
		public readonly bool Overwrite;
		public override FileList Sources { get { return (sources); }}
		
		// constructor
		public InstallCommand (string name, string dest, string mode, bool overwrite)
		{
			if (name == null)
				throw new ArgumentNullException ("name");
			if (dest == null)
				throw new ArgumentNullException ("dest");
			
			Name = name;
			Dest = dest;
			Mode = mode;
			if (Mode == null)
				Mode = "0644";
			Overwrite = overwrite;
			
			sources.Add (Name);
		}
		
		// constructor form an XmlElement
		public InstallCommand (XmlElement installelement)
		{
			if (installelement == null)
				throw new ArgumentNullException ("installelement");
			
			// Get the source
			Name = installelement.GetAttribute ("name");
			// Get the destination
			Dest = installelement.GetAttribute ("dest");
			// Get the options
			Mode = installelement.GetAttribute ("mode");
			if (Mode == "")
				Mode = "0644";
			if (installelement.GetAttribute ("overwrite") == "no")
				Overwrite = false;
			else
				Overwrite = true;
			
			sources.Add (Name);
		}
		
		public override BashCommand ToBashCommand (bool verbose)
		{
			string command;
			
			if (((string)Name)[0] == System.IO.Path.DirectorySeparatorChar)
				command = "install -D -m " + Mode + " " + Name + " " + Dest;
			else
				command = "install -D -m " + Mode + " $PACKAGESOURCEDIR/" + Path.GetFileName (Name) + " " + Dest;
			
			if (Overwrite)
				return ((new SimpleCommand (command, true)).ToBashCommand (verbose));
			else
				return ((new SimpleCommand ("[ -e " + Dest + " ] || " + command, true)).ToBashCommand (verbose));
		}
		
		private FileList sources = new FileList ();
	}
}
