// Local.cs
//
// Process the upkg.conf XML file [SINGLETON]
//
// Copyright (C) 2004-2005 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.Collections;
using System.IO;
using System.Xml;
using System.Net;

namespace Upkg
{
	public class Local
	{
		// holds the repositories
		public static readonly RepositoryList Repositories = new RepositoryList();
		// holds the info for the default package
		public static string RootPackageName = null;
		public static string RootPackageTag
		{
			get
			{
				return (rootpackagetag);
			}
			set
			{
				if (value == null || value == "")
					rootpackagetag = "default";
				else
					rootpackagetag = value;
			}
		}
		// boostrap
		public static bool Bootstrap = false;
		// holds the branch we're using
		public static Branch Branch;
		// specifies the architecture of the building maching
		public static string HostArchitecture = "x86";
		// specifies the architecture of the target maching
		public static string Architecture = "x86";
		// print file to stdout (instead writing to file)
		public static bool Stdout = false;
		// inidicates whether we should print verbosely to the console
		public static bool Verbose = false;
		public static bool AllowSourceBuild = true;
		public static bool AllowBinaryInstall = true;
		// holds the local cachedir
		public static string CacheDir = null;
		// is cachedir relative to chrootdir when bootstrapping?
		public static bool CacheDirChroot = false;
		// would we like to have the specs cached
		public static bool EnableCaching = true;
		// ignore .select files and don't remove any packages
		// used by upkg upgrade script generator
		public static bool IgnoreSelection = false;
		// proxy acessor
		public static string Proxy
		{
			get
			{
				return proxyUrl;
			}
			set
			{
				proxyUrl = value;
				proxyObj = new WebProxy(value);
			}
		}
		// proxy URL (if any)
		private static string proxyUrl = null;
		// single web proxy object
		private static WebProxy proxyObj = null;
		
		// singleton constructor
		static Local ()
		{
			// create the xmldocument
			XmlDocument xmldoc = new XmlDocument();
			// load the file
			xmldoc.Load (new XmlValidatingTextReader (file, new string[] {Config.SHAREDIR + "/upkg.xsd"}));
			
			// go through all nodes
			foreach (XmlNode node in xmldoc.DocumentElement.ChildNodes)
			{
				XmlElement element = node as XmlElement;
				// ignore non-element nodes (such as comments)
				if (element == null)
					continue;
				
				// Check all elements which could occur
				if (element.Name == "cachedir")
				{
					CacheDir = element.InnerText;
					// create the cachedir
					if (!Directory.Exists (CacheDir))
						Directory.CreateDirectory (CacheDir);
						
					CacheDirChroot = (element.GetAttribute ("chroot") == "yes");
				}
				else if (element.Name == "repositories")
				{
					// add all specified and reachable repositories to our list
					foreach (XmlNode repo in element.ChildNodes)
					{
						// Check if we have an XmlElement
						if (!(repo is XmlElement))
							continue;
						
						Repositories.Add (new Repository (repo.InnerText));
					}
				}
				else if (element.Name == "package")
				{
					RootPackageName = element.GetAttribute ("name");
					// get the default package tag
					RootPackageTag = element.GetAttribute ("tag");
				}
				else if (element.Name == "branch")
				{
					Branch = node.InnerText;					
				}
				else if (element.Name == "arch")
				{
					Architecture = node.InnerText;					
					HostArchitecture = Architecture;
				}
				else if (element.Name == "proxy")
				{
					Proxy = node.InnerText;
				}
			}
			
			// Check all needed values are set
			if (Repositories.Count == 0 || CacheDir == null)
				throw new ApplicationException("Local: could not fetch enough info.");
				
			// create a chachedir for each repo
			foreach (Repository repo in Repositories)
			{
				if (!Directory.Exists (RepoCacheDir (repo.Url)))
					Directory.CreateDirectory (RepoCacheDir (repo.Url));
			}
		}
		
		public static string RepoCacheName (string repo)
		{
			return (repo.Replace ("http://", "").Replace ("https://", "").Replace ('/', '_'));
		}
		
		public static string RepoCacheDir (string repo)
		{
			return (Path.Combine (CacheDir, RepoCacheName (repo)));
		}
		
		public static HttpWebRequest CreateHttpWebRequest(string url)
		{
			HttpWebRequest req = (HttpWebRequest) WebRequest.Create(url);
			if (proxyUrl != null)
			{
				req.Proxy = proxyObj;
			}
			return req;
		}
		
		// holds the path to upkg.conf
		protected const string file = Config.SYSCONFDIR + "/upkg.conf";
		
		private static string rootpackagetag;
	}

	public enum Mode
	{
		Build,
		Install
	}
}
