// Package.cs
//
// Process Package XML Files and gather all sort of Information from it
//
// Copyright (C) 2004 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.Xml;
using System.Collections;
using System.IO;

namespace Upkg
{
	public class Package
	{
		// holds the name of the package
		public readonly string Name;
		// one-line description of the package
		public readonly string Description;
		// holds the release specifications for all tags
		// Hashtable <string, Branches <Branch, ReleaseSpecification>>
		public readonly Hashtable Tags = new Hashtable ();
		// holds the xmldoc
		public readonly XmlSpecification Specification;
		
		// constructor: protected as this will only be called by GetPackage to ensure there is only one package object per package name
		protected Package (string name, bool allowZombies)
		{
			// Check the parameters
			if (name == null)
				throw new ArgumentNullException("name");
			else if (name == "")
				throw new ArgumentException("empty: name");
				
			// Set the main values
			Name = name;
			
			if (Local.Verbose)
				Console.Error.WriteLine ("Processing " + Name + " package...");
			
			Specification = new XmlSpecification (name + ".xml");
			if (Specification.DocumentElement == null)
			{
				if (!allowZombies)
					throw (new PackageNotFoundException (name));
				
				return;
			}

			// Check if the package name is equal to the filename
			if (Specification.DocumentElement.GetAttribute ("name") != Name)
				throw new ApplicationException("Package: Package name does not match its filename.");
			
			// Process each release specification
			foreach (XmlElement releaseElement in Specification.GetElementsByTagName ("release"))
			{
				// Ignore releases for other architectures
				if (!Architecture.IsActive (releaseElement.GetAttribute ("arch")))
					continue;
				
				ReleaseSpecification release = new ReleaseSpecification (this, releaseElement);
				if (!Tags.ContainsKey (release.Tag))
					Tags[release.Tag] = new Branches ();
				((Branches)Tags[release.Tag]).Add (release.Branch, release);
			}
		}
		
		public static Package GetPackage (string name)
		{
			return (GetPackage (name, false));
		}
		
		public static Package GetPackage (string name, bool allowZombies)
		{
			if (!Packages.ContainsKey (name))
				Packages[name] = new Package (name, allowZombies);

			return (Packages[name] as Package);
		}
		
		public static bool HasPackage (string name)
		{
			if (name == null)
				throw (new ArgumentNullException ("name"));
			
			return (Packages.ContainsKey (name));
		}
		
		// clear all processed packages
		public static void Flush ()
		{
			Packages.Clear();
		}
		
		protected static Hashtable Packages = new Hashtable ();
		
		// returns the hashcode out of the package name
		public override int GetHashCode ()
		{
			return (Name.GetHashCode());
		}
	}
}
