// PatchCommand.cs
//
// Holds all information a patch command has
//
// Copyright (C) 2004 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.Collections;
using System.IO;
using System.Xml;

namespace Upkg
{
	class PatchCommand : Command
	{
		// holds the name
		public readonly ResolvableString Name;
		// holds the compression of the patch
		public readonly string Compression;
		// holds the strip value
		public readonly uint Strip;
		// Sources needed
		public override FileList Sources { get { return (sources); }}
		// holds the dir where the patch will be applied in
		protected readonly string Patchdir;
		
		// constructor
		public PatchCommand (string name, string compression, uint strip, string patchdir)
		{
			if (name == null)
				throw new ArgumentNullException ("name");
			if (compression == null)
				throw new ArgumentNullException ("compression");
				
			Name = name;
			Compression = compression;
			Strip = strip;
			Patchdir = patchdir;
			
			// Add the file to the filelist
			filename = name + ".patch";
			if (Compression != "plain")
				filename += "." + Compression;
			sources.Add (filename);
		}
		
		// constructor with XmlElement
		public PatchCommand (XmlElement patchtype, string patchdir)
		{
			if (patchtype == null)
				throw new ArgumentNullException ("patchtype");
			
			Patchdir = patchdir;
			// Get the strip value
			if (patchtype.Attributes.GetNamedItem("strip") != null)
				Strip = uint.Parse(patchtype.Attributes.GetNamedItem("strip").InnerText);
			else
				Strip = 1;
			// Get the compression
			if (patchtype.Attributes.GetNamedItem("compression") != null)
				Compression = patchtype.Attributes.GetNamedItem("compression").InnerText;
			else
				Compression = Global.Compression;
			// Get the filename
			string name = patchtype.Attributes.GetNamedItem("name").InnerText;
			Name = name;
			// Add the file to the filelist
			filename = name + ".patch";
			if (Compression != "plain")
				filename += "." + Compression;
			sources.Add (filename);
		}
		
		// the shell func
		public override BashCommand ToBashCommand (bool verbose)
		{
			string command;
				
			// check for if we need to change to the srcdir
			if (Patchdir != null)
				command = "patch -d ../" + Patchdir + " -Np";
			else
				command = "patch -Np";
				
			// Create the command
			if (Compression == "bz2")
				command = "bzip2 -dc $PACKAGESOURCEDIR/" + Path.GetFileName (filename) + " | " + command + Strip;
			else if (Compression == "gz")
				command = "gzip -dc $PACKAGESOURCEDIR/" + Path.GetFileName (filename) + " | " + command + Strip;
			else if (Compression == "xz")
				command = "xz -dc $PACKAGESOURCEDIR/" + Path.GetFileName (filename) + " | " + command + Strip;
			else if (Compression == "zst")
				command = "zstd -dc $PACKAGESOURCEDIR/" + Path.GetFileName (filename) + " | " + command + Strip;
			else if (Compression == "plain")
				command = command + Strip + " -i $PACKAGESOURCEDIR/" + Path.GetFileName (filename);
			else
				throw new ApplicationException("PatchCommand: Unsupported patch compression requested (" + Compression + ")");
				
			return ((new SimpleCommand (command, true)).ToBashCommand (verbose));
		}
		
		private FileList sources = new FileList ();
		private ResolvableString filename;
	}
}
