// ReleaseList.cs
//
// Holds a list of releases
//
// Copyright (C) 2004 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>


using System;
using System.Collections;
using System.IO;
using Mono.Unix;
using System.Text;
using System.Collections.Specialized;

namespace Upkg
{
	public class ReleaseList : ArrayList
	{
		// add specified release
		public void AddRelease (string packageName, string tag, bool allowZombies)
		{
			Package package;
			Branches tagBranches;
			ReleaseSpecification release;
		
			if (packageName == null)
				throw (new ArgumentNullException ("packageName"));
			
			if (tag == null)
				tag = "default";
			
			package = Package.GetPackage (packageName, allowZombies);

			tagBranches = (Branches) package.Tags[tag];
			if (tagBranches == null)
			{
				tagBranches = new Branches ();
				package.Tags[tag] = tagBranches;
			}

			release = (ReleaseSpecification) tagBranches.Current;
			if (release == null)
			{
				release = new ReleaseSpecification (package, tag, Local.Branch);
				tagBranches.Add (Local.Branch, release);
			}

			// add the corresponding release to the list
			Add (release);
		}
	
		public void AddRelease (string packageName, string tag)
		{
			AddRelease (packageName, tag, false);
		}

		// add all releases corresponding to the user's package selection
		public void AddSelectedReleases ()
		{
			NameValueCollection info;
			string path;
			
			Settings settings = ((BranchSpecification) Global.Branches.Current).Settings;
			
			path = (string) settings.Variables["$STATEDIR"];
			if (Local.Bootstrap)
				path = (string) settings.Variables["$CHROOTDIR"] + path;
			
			path = Path.Combine (path, "packages");
			
			// if the statedir doesn't exist, the package selection must be empty
			if (!Directory.Exists (path))
				return;
			
			// iterate over all files ending with .select
			foreach (string file in Directory.GetFiles (path, "*.select"))
			{
				info = ColonSeparatedData.Parse (File.OpenRead (file));
				AddRelease (info["Package"], info["Tag"], true);
			}
			
			// backward compatibility
			if (Local.RootPackageName != null)
				AddRelease (Local.RootPackageName, Local.RootPackageTag, true);
			AddKeepReleases (path);
		}
		
		// add releases marked with a .keep file
		// (originating from old Upkg version)
		void AddKeepReleases (string path)
		{
			NameValueCollection info;
			
			// go through all files ending with .keep
			foreach (string file in Directory.GetFiles (path, "*.keep"))
			{
				// follow the link of the UniqueName of the package to its VersionedName and append .info to get its info file; open and parse it
				string symlink = file.Replace (".keep", null);
				if (!File.Exists (symlink))
				{
					// we've found a keep file without corresponding info file
					// this could come from an older version of upkg-remove
					Console.Error.WriteLine ("Warning: Ignoring stale keep file {0}", file);
					continue;
				}
				// we only need keep or select file
				if (File.Exists (file.Replace (".keep", ".select"))) {
					continue;
				}
				info = ColonSeparatedData.Parse (File.OpenRead (UnixPath.GetRealPath (symlink) + ".info"));
				AddRelease (info["Package"], info["Tag"], true);
			}
		}
	}
}
