// ReleaseSpecification.cs
//
// Holds the specification of a package release
//
// Copyright (C) 2004-2005 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.Collections;
using System.Xml;
using System.IO;

namespace Upkg
{
	public class ReleaseSpecification : Specification
	{
		// Packages get marked virtual when they only consist of deps and possibly postbuild commands
		public bool Virtual
		{
			get
			{
				if (!processed)
					throw (new InvalidOperationException ("Instance not yet processed."));

				return (_virtual);
			}
		}
		private bool _virtual = true;
	
		// constructor
		public ReleaseSpecification (Package package, XmlElement SpecElement)
		{
			if (package == null)
				throw (new ArgumentNullException ("package"));
			if (SpecElement == null)
				throw (new ArgumentNullException ("SpecElement"));
				
			this.SpecElement = SpecElement;

			Package = package;
			BranchSpecification current_branch = (BranchSpecification) Global.Branches.Current;
			Tag = current_branch.Settings.ResolveVariables (SpecElement.GetAttribute ("tag"));
			if (Tag == "")
				Tag = "default";
			Branch = Local.Branch;
			Version = SpecElement.GetAttribute ("version");

			string revString = SpecElement.GetAttribute ("revision");
			if (revString != "")
				Revision = Int32.Parse (revString);

			Hash = SpecElement.GetAttribute ("hash");
		}
		
		// constructor for zombie releases (releases still selected but already removed from repository)
		public ReleaseSpecification (Package package, string tag, Branch branch)
		{
			if (package == null)
				throw (new ArgumentNullException ("package"));
				
			Package = package;
			if (tag == null || tag == "")
				Tag = "default";
			else
				Tag = tag;
			Branch = branch;
			
			Zombie = true;
		}
		
		// finish processing the xmlelement
		// will be called by Branches class
		public override void Process ()
		{
			// is the object well enough initialized to be processed?
			if (SpecElement == null)
				throw (new ApplicationException ("SpecElement is not initialized"));
			
			if (!processed)
			{
				// mark as processed before anything else so we don't end in an endless loop
				processed = true;
				
				// get the BranchSpecification of the parent scope, so we can join them
				BranchSpecification parentSpec = (BranchSpecification) Upkg.Tag.GetTag (Tag).Branches.Current;
				if (parentSpec == null)
					parentSpec = (BranchSpecification) Upkg.Architecture.GetArchitecture (Local.Architecture).Branches.Current;
				if (parentSpec == null)
					parentSpec = (BranchSpecification) Global.Branches.Current;
				
				// process the settings element separately as the other elements may depend on this
				Settings = ProcessSettings (parentSpec);

				// Add The package info to the variables
				Settings.Variables.Add ("$NAME", Package.Name);
				Settings.Variables.Add ("$TAG", Tag);
				Settings.Variables.Add ("$VERSION", Version);
				Settings.Variables.Add ("$REVISION", Revision.ToString ());
				Settings.Variables.Add ("$VERSIONEDNAME", VersionedName);
				Settings.Variables.Add ("$BUILDNAME", VersionedName);
				Settings.Variables.Add ("$PACKAGEBRANCH", Branch.ToString ());
				Settings.Variables["$CACHEDIR"] = Local.RepoCacheDir (Package.Specification.Repository.Url);
				Settings.Variables.Add ("$PACKAGESOURCEDIR", Settings.ResolveVariables (Path.Combine (Local.RepoCacheDir (Package.Specification.Repository.Url), "sources/$NAME")));
				Settings.Variables.Add ("$PACKAGEREPOSITORY", Package.Specification.Repository.Url);
				
				PostInst.AddRange (parentSpec.PostInst);
				PreRm.AddRange (parentSpec.PreRm);
				Strip.AddRange (parentSpec.Strip);

				foreach (XmlNode node in SpecElement.ChildNodes)
				{
					XmlElement element = node as XmlElement;
					// ignore non-element nodes (as for example comments)
					if (element == null)
						continue;
					
					// process the childs of <release>
					switch (element.Name)
					{
					case "deps":
						ProcessDeps (element);
						break;
					case "preinst":
						PreInst.Process (element);
						_virtual = false;
						break;
					case "build":
						ProcessBuild (element, parentSpec);
						_virtual = false;
						break;
					case "postinst":
						PostInst.Process (element);
						_virtual = false;
						break;
					case "prerm":
						PreRm.Process (element);
						_virtual = false;
						break;
					case "config":
						ProcessConfig (element);
						_virtual = false;
						break;
					case "postbuild":
						PostBuild.Process (element);
						break;
					}
				}

				if (BuildScript != null && BuildScript.Sources != null)
				{
					BuildSources.AddRange (BuildScript.Sources);
				}
				
				// add ourself to the releaselist
				ReleaseList[UniqueName] = this;
			}
		}

		private void ProcessDeps (XmlElement depsElement)
		{
			foreach (XmlNode node in depsElement.ChildNodes)
			{
				XmlElement element = node as XmlElement;
				// ignore non-element nodes (as for example comments)
				if (element == null)
					continue;
				
				string name = element.GetAttribute ("name");
				BranchSpecification current_branch = (BranchSpecification) Global.Branches.Current;
				string tag = current_branch.Settings.ResolveVariables (element.GetAttribute ("tag"));
				if (tag == "")
					tag = "default";

				// Ignore deps for other architectures
				if (!Architecture.IsActive (element.GetAttribute ("arch")))
					continue;
				
				Package pkg = Package.GetPackage (name);
				
				if (pkg.Tags[tag] == null || ((Branches)pkg.Tags[tag]).Current == null)
					throw (new ApplicationException("Release '" + UniqueName + "' has an invalid/unknown dependency '" + GetUniqueName (name, tag) + "'"));
				Deps.Add ((ReleaseSpecification)((Branches)pkg.Tags[tag]).Current);
			}
		}
		
		private void ProcessBuild (XmlElement buildElement, BranchSpecification parentSpec)
		{
			if (buildElement.GetAttribute ("strip") == "no")
				BuildStrip = false;
		
			if (parentSpec.IgnoreFiles != null)
				BuildIgnoreFiles.AddRange (parentSpec.IgnoreFiles);

			foreach (XmlNode node in buildElement.ChildNodes)
			{
				XmlElement element = node as XmlElement;
				// ignore non-element nodes (as for example comments)
				if (element == null)
					continue;
				
				// process the childs of <release>
				// FIXME: add and ignore have to be joined with parent scope (global/tag)
				switch (element.Name)
				{
				case "sources":
					BuildSources.Process (element);
					break;
				case "script":
					if (parentSpec.Init != null)
					{
						BuildScript.AddRange (parentSpec.Init);
					}
					BuildScript.Process (element);
					break;
				case "add":
					BuildAddFiles.Process (element);
					break;
				case "ignore":
					BuildIgnoreFiles.Process (element);
					break;
				}
			}
		}
		
		// Processes all config childs and adds the neccesary commands to the command lists
		protected void ProcessConfig (XmlElement e)
		{
			int i = 1;
			Script configremove = new Script ();
			
			// add a local array to store eventual configfile deletes
			if (!Local.Bootstrap)
				BuildScript.Insert (0, new SimpleCommand ("declare -a configfiles", true));
			
			// go through all childs
			foreach (XmlNode node in e.ChildNodes)
			{
				XmlElement element = node as XmlElement;
				// ignore non-element nodes (as for example comments)
				if (element == null)
					continue;

				ConfigFile file = new ConfigFile (element);
				ConfigFiles.Add (file);
				
				//add a check if the configfile exists
				if (!Local.Bootstrap)
					BuildScript.Insert (i, new SimpleCommand ("[ -e " + file.Dest + " ] || configfiles[" + i.ToString() + "]=true", true));
				
				// copy the file to default
				if (file.Name != null)
					BuildScript.Add (new InstallCommand (file.Name, file.Default, file.Mode, true));
				if (Local.Bootstrap)
					BuildScript.Add (new SimpleCommand ("rm -f " + file.Dest, true));
				
				// copy the configfile to the final dest
				PostInst.Insert (0, new InstallCommand (file.Default, file.Dest, file.Mode, false));
				
				// on removal delete the config file if it was not changed
				PreRm.Add (new SimpleCommand ("cmp -s " + file.Default + " " + file.Dest + " && rm -f " + file.Dest, false));
				
				// add the possible removal
				if (!Local.Bootstrap)
					configremove.Add (new SimpleCommand ("[ \"${configfiles[" + i.ToString() + "]}\" != \"true\" ] || rm -f " + file.Dest, true));
				
				i++;
			}
			
			BuildScript.AddRange (configremove);
		}
		
		// holds the name of the release
		public string Name
		{
			get
			{
				string basename = UniqueName + "-" + Version;
				if (Revision > 0)
					return basename + "-" + Revision.ToString();
				else
					return basename;
			}
		}
		public override string ToString ()
		{
			return (Name);
		}
		// holds the name of the package
		public readonly Package Package;
		// holds the tag of the release
		public readonly string Tag;
		// Unique Name of the package
		public string UniqueName { get { return GetUniqueName (Package.Name, Tag); } }
		// returns the package version this release refers to
		public readonly string Version;
		// returns the hash of the package contents
		public readonly string Hash;
		// holds the revision number
		public readonly int Revision;
		// zombie release (still selected but already removed from repository)
		public readonly bool Zombie = false;
		// Versioned package name: name-tag-arch-toolchain-version-revision
		public string VersionedName
		{
			get
			{
				if (Hash != null && Hash != "")
					return (UniqueName + "-" + Local.Architecture + "-" + Version + "-" + Hash);
				else
					return (UniqueName + "-" + Local.Architecture + "-" + (Global.Branches.Current as BranchSpecification).Settings.Variables["$TOOLCHAIN"] + "-" + Version + "-" + Revision.ToString());
			}
		}
		// Binary name: VersionedName + Branch
		public string BinaryName
		{
			get
			{
				if (Hash != null && Hash != "")
					return (VersionedName + ".tar." + Global.Compression);
				else
					return (VersionedName + "-" + BinaryBranch + ".tar." + Global.Compression);
			}
		}
		// holds the branch, this release is in
		public readonly Branch Branch;
		// holds max (Global.UpperToolchainBranchBound, Branch)
		public Branch BinaryBranch
		{
			get
			{
				if (Branch <= Global.UpperToolchainBranchBound)
					return (Branch);
				else
					return (Global.UpperToolchainBranchBound);
			}
		}
		// returns the Settings
		public Settings Settings;
		// returns the files used
		public readonly FileList BuildSources = new FileList ();
		// holds the file from the add section
		public readonly FileList BuildAddFiles = new FileList ();
		// holds the file from the ignore section
		public readonly FileList BuildIgnoreFiles = new FileList ();
		// returns the list of all releases this release depends on [ReleaseSpecification]
		public readonly ArrayList Deps = new ArrayList ();
		// holds the build commands [Command]
		public readonly Script BuildScript = new Script();
		// holds the preinstallation commands [Command]
		public readonly Script PreInst = new Script();
		// holds the postinstallation commands [Command]
		public readonly Script PostInst = new Script();
		// holds the preremove commands [Command]
		public readonly Script PreRm = new Script();
		// holds the postbuild commands [Command]
		public readonly Script PostBuild = new Script();
		// holds the strip commands [Command]
		public readonly Script Strip = new Script();
		// holds the files from the config section
		public readonly ArrayList ConfigFiles = new ArrayList ();
		// Returns whether we want to have our package stripped or not
		public bool BuildStrip = true;
		// is this release part of the package selection (.select file resp. specified via command line)
		public bool Selected = false;
		
		public static readonly Hashtable ReleaseList = new Hashtable();
		
		// returns the hashcode out of the name AND tag strings
		public override int GetHashCode ()
		{
			return (UniqueName.GetHashCode());
		}
		
		public static string GetUniqueName (string name, string tag)
		{
			return (name + (tag != "default" ? "-" + tag : null));
		}
				
		private bool processed = false;
	}
}
