// Script.cs
//
// Processes a scriptType Element
//
// Copyright (C) 2004 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.Collections;
using System.Xml;

namespace Upkg
{
	public class Script : ArrayList, ICommand
	{
		public virtual FileList Sources { get { return (sources); }}
		//protected XmlElement ScriptElement;

		// constructor
		public Script () : base ()
		{
		}
		
		// process the elements
		public void Process (XmlElement scripttype)
		{
			// check the arguments
			if (scripttype == null)
				throw new ArgumentNullException("scripttype");
			
			foreach (XmlNode node in scripttype.ChildNodes)
			{
				// Check if we have an XmlElement
				if (node.NodeType != XmlNodeType.Element)
					continue;
					
				CallProcessorForElement ((XmlElement)node);
			}
		}
		
		// call the correct processor for a given XmlElement
		protected virtual void CallProcessorForElement (XmlElement command)
		{
			if (command == null)
				throw new ArgumentNullException ("command");
			
			Command c = null;
			
			switch (command.Name)
			{
				case "archive":
					c = new ArchiveCommand (command);
					break;
				case "install":
					c = new InstallCommand (command);
					break;
				case "patch":
					c = new PatchCommand (command, null);
					break;
				case "cmd":
					c = new SimpleCommand (command);
					break;
				case "text":
					c = new TextCommand (command);
					break;
				default:
					throw new ApplicationException ("Script: unknown script child " + command.Name);
			}
			
			Add (c);
		}
		
		// implementation
		public virtual BashCommand ToBashCommand (bool verbose)
		{
			BashList result = new BashList ();
			
			foreach (Command c in this)
			{
				result.Add (c.ToBashCommand (verbose));
			}
			
			return (result);
		}
		
		// helper
		protected void AddSources (FileList sources)
		{
			if (sources != null)
				this.sources.AddRange (sources);
		}
		
		// overwrite add
		public override int Add (object obj)
		{
			if (!(obj is Command))
				throw new ArgumentException ("Type missmatch: obj");
			
			Command c = (Command)obj;	
			
			AddSources (c.Sources);
			return (base.Add (obj));
		}
		
		private FileList sources = new FileList ();
	}
}
