// Package.cs
//
// Helper class to process the Settings Type inside XML documents
//
// Copyright (C) 2004-2005 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.Xml;
using System.Collections;

namespace Upkg
{
	public class Settings
	{
		// the variables [string,string]
		public Hashtable Variables { get { return (variables); } }
		// the environment variables [string,string]
		public Hashtable Environment { get { return (environment); } }
		// staging
		public bool Staging
		{
			get
			{
				if (staging == Bool3.Unset)
					throw (new ApplicationException ("Settings.Staging unset"));
				return (staging == Bool3.True);
			}
		}
		// allow binary installation or is this a source-only package?
		public bool AllowBinaryInstall
		{
			get
			{
				if (allowBinaryInstall == Bool3.Unset)
					throw (new ApplicationException ("Settings.AllowBinaryInstall unset"));
				return (allowBinaryInstall == Bool3.True && register == Bool3.True && (!Staging || Local.Architecture == Local.HostArchitecture));
			}
		}
		
		// shall this pacckage be registerd in the package database?
		public bool Register
		{
			get
			{
				if (register == Bool3.Unset)
					throw (new ApplicationException ("Settings.Register unset"));
				return (register == Bool3.True);
			}
		}
		
		// shall this pacckage be set on hold?
		public bool Keep
		{
			get
			{
				if (keep == Bool3.Unset)
					throw (new ApplicationException ("Settings.Keep unset"));
				return (keep == Bool3.True);
			}
		}
		
		// Default compression format
		public string Compression { get { return compression; } }
		
		public Settings (XmlElement settingsElement) : this (settingsElement, null)
		{
		}
		
		// Process the elemets insite a SettingsType element
		public Settings (XmlElement settingsElement, Settings parent)
		{
			if (settingsElement == null && parent == null)
				throw (new ArgumentException ("settingsElement and/or parent have to be not null"));

			// import variables from the parent context
			if (parent != null)
			{
				variables = (Hashtable)parent.Variables.Clone ();
				environment = (Hashtable)parent.Environment.Clone ();
				staging = (parent.Staging ? Bool3.True : Bool3.False);
				allowBinaryInstall = (parent.AllowBinaryInstall ? Bool3.True : Bool3.False);
				register = (parent.Register ? Bool3.True : Bool3.False);
				keep = (parent.Keep ? Bool3.True : Bool3.False);
				compression = parent.Compression;
			}
			else
			{
				// if there is no parent context we just start with empty hashtables
				variables = new Hashtable ();
				environment = new Hashtable ();
				
				variables["$ARCH"] = Local.Architecture;
			}

			if (settingsElement == null)
				return;

			// go through all childs of the settings
			foreach (XmlNode n in settingsElement.ChildNodes)
			{
				XmlElement e = n as XmlElement;
				// ignore non-element nodes such as comments
				if (e == null)
					continue;

				// Go through all possible elements
				if (e.Name == "builddir" || e.Name == "chrootdir" || e.Name == "statedir" || e.Name == "toolchain")
					Variables["$" + e.Name.ToUpper()] = ResolveVariables (e.InnerText);
				// treat staging special
				else if (e.Name == "staging")
				{
					if (e.InnerText == "yes")
						staging = Bool3.True;
					else if (e.InnerText == "no")
						staging = Bool3.False;
					else
						throw new ApplicationException("Settings: Element 'staging' has an invalid value");
				}
				// treat binary special
				else if (e.Name == "binary")
				{
					if (e.InnerText == "yes")
						allowBinaryInstall = Bool3.True;
					else if (e.InnerText == "no")
						allowBinaryInstall = Bool3.False;
					else
						throw new ApplicationException("Settings: Element 'binary' has an invalid value");
				}
				// register
				else if (e.Name == "register")
				{
					if (e.InnerText == "yes")
						register = Bool3.True;
					else if (e.InnerText == "no")
						register = Bool3.False;
					else
						throw new ApplicationException("Settings: Element 'register' has an invalid value");
				}
				// keep
				else if (e.Name == "keep")
				{
					if (e.InnerText == "yes")
						keep = Bool3.True;
					else if (e.InnerText == "no")
						keep = Bool3.False;
					else
						throw new ApplicationException("Settings: Element 'keep' has an invalid value");
				}
				// compression format
				else if (e.Name == "compression")
				{
					compression = e.InnerText;
					if (compression != "gz" && compression != "bz2" && compression != "xz" && compression != "zst")
						throw new ApplicationException("Settings: Element 'compression' has an unsupported value: " + compression);
				}
				// treat non specific variables special
				else if (e.Name == "variable")
				{
					Variables["$" + e.GetAttribute ("name").ToUpper()] = ResolveVariables (e.InnerText);
				}
				// treat environment special
				else if (e.Name == "environment")
				{
					Environment[e.GetAttribute ("name")] = new ResolvableString (e.InnerText);
				}
			}

			// we're in global context, so ensure we have all mandatory variables set
			if (parent == null)
				CheckDefaultValues ();
		}
		
		// Ensures that all vital settings are set to at least their default values
		protected void CheckDefaultValues ()
		{
			// Check all the mandatory variables and set them to a default value if not
			if (!Variables.Contains("$BUILDDIR"))
				Variables["$BUILDDIR"] = "/usr/src";
			if (!Variables.Contains("$CACHEDIR"))
				Variables["$CACHEDIR"] = "/var/cache/upkg";
			if (!Variables.Contains("$STATEDIR"))
				Variables["$STATEDIR"] = "/var/lib/upkg";
			if (!Variables.Contains("$CHROOTDIR"))
				Variables["$CHROOTDIR"] = "/upkg";
			
			// Define the $REPOSITORIES variable
			Variables["$REPOSITORIES"] = "";
			bool first = true;
			foreach (Repository repo in Local.Repositories)
			{
				if (!first)
					Variables["$REPOSITORIES"] += " ";
				else
					first = false;
				
				Variables["$REPOSITORIES"] += repo.Url;
			}

			Variables["$CACHEDIR"] = Local.CacheDir;

			// Default staging to false
			if (staging == Bool3.Unset)
				staging = Bool3.False;
			
			// Default binary to true
			if (allowBinaryInstall == Bool3.Unset)
				allowBinaryInstall = Bool3.True;
				
			// Default register to true
			if (register == Bool3.Unset)
				register = Bool3.True;
			
			// Default keep to false
			if (keep == Bool3.Unset)
				keep = Bool3.False;
			
			// Default to zstd compression
			if (compression == null || compression == "")
				compression = "zst";
		}

		// Takes a string and resolves all known variables
		public string ResolveVariables (string s)
		{
			string olds;
			do
			{
				olds = s;
				foreach (string key in Variables.Keys)
				{
					s = s.Replace (key, (string) Variables[key]);
				}
			}
			while (s != olds);
			
			return (s);
		}
		
		// Activate this settings instance
		public void Activate ()
		{
			current = this;
		}
		
		// Deactivate current settings
		public static void Deactivate ()
		{
			current = null;
		}

		public static Settings Current { get { return (current); } }

		// the internal variables collection
		protected Hashtable variables;
		// the internal environment variables collection
		protected Hashtable environment;
		// staging
		protected Bool3 staging = Bool3.Unset;
		// installable
		protected Bool3 allowBinaryInstall = Bool3.Unset;
		// register
		protected Bool3 register = Bool3.Unset;
		// hold
		protected Bool3 keep = Bool3.Unset;
		// compression format
		protected string compression;
		
		protected static Settings current = null;
		
	}
}
