// InstallCommand.cs
//
// Holds all information a install command element has
//
// Copyright (C) 2005 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>


using System;
using System.Collections;
using System.IO;
using System.Xml;

namespace Upkg
{
	class TextCommand : Command
	{
		public readonly ResolvableString Text;
		public readonly ResolvableString Dest;
		public readonly string Mode;
		public readonly bool Append;
		public readonly bool ExpandEnvironment;
		
		// constructor
		public TextCommand (string dest, string text, string mode, bool append, bool expand_environment)
		{
			if (dest == null)
				throw new ArgumentNullException ("dest");
			
			Dest = dest;
			Mode = mode;
			if (Mode == null)
				Mode = "0644";
			Append = append;
			Text = text;
			ExpandEnvironment = expand_environment;
		}
		
		// constructor form an XmlElement
		public TextCommand (XmlElement textelement)
		{
			if (textelement == null)
				throw new ArgumentNullException ("textelement");
			
			Dest = textelement.GetAttribute ("dest");
			// Get the options
			Mode = textelement.GetAttribute ("mode");
			if (Mode == "")
				Mode = "0644";
			if (textelement.GetAttribute ("append") == "yes")
				Append = true;
			else
				Append = false;
				
			Text = textelement.InnerText;
			
			if (textelement.GetAttribute ("environment") == "expand")
				ExpandEnvironment = true;
			else
				ExpandEnvironment = false;
		}
		
		public override BashCommand ToBashCommand (bool verbose)
		{
			BashList outer = new BashList ();
			BashList inner = new BashList ();
			string cat_command, text_part;
			
			cat_command = "cat >";
			
			if (Append)
				cat_command += ">";
			
			if (ExpandEnvironment)
				cat_command += " " + Dest + "<< EOF";
			else
				cat_command += " " + Dest + "<< \"EOF\"";
			
			text_part = Text + "\nEOF";
			
			if (verbose)
				outer.Add (new BashCommand ("echo \" [CHECK] " + BashScriptWriter.QuoteString (cat_command + "\n" + text_part) + "\""));
				
			outer.Add (new SimpleCommand (cat_command, true).ToBashCommand (false));
			inner.Preformatted = true;
			inner.Add (new BashCommand (text_part));
			outer.Add (inner);
			outer.Add (new SimpleCommand ("chmod " + Mode + " " + Dest, true).ToBashCommand (verbose));
			
			return (outer);
		}
	}
}
