// Specification.cs
//
// represents a specification xml file
// also implements caching of xml specs
//
// Copyright (C) 2004 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.Xml;
using System.Xml.Schema;
using System.Collections;
using System.Net;
using System.IO;

namespace Upkg
{
	public class XmlSpecification : XmlDocument
	{
		public readonly Repository Repository;
		
		public XmlSpecification (string file)
		{
			string cachefilepath = null;
			string repofilepath = null;
			string chosenfilepath = null;
			
			// fetch the most important version of the file
			foreach (Repository repo in Local.Repositories)
			{
				if (Local.EnableCaching)
					cachefilepath = Local.RepoCacheDir (repo.Url) + "/specs/" + file;
					
				repofilepath = repo.Url + "/specs/" + file;
				
				if (!repo.Remote)
				{
					if (File.Exists (repofilepath))
						chosenfilepath = repofilepath;
				}
				else
				{
					if (Local.EnableCaching)
					{
						// create a specs folder
						if (!Directory.Exists (Local.RepoCacheDir (repo.Url) + "/specs"))
							Directory.CreateDirectory (Local.RepoCacheDir (repo.Url) + "/specs");
						// examine an existing copy of the file in the cache
						if (File.Exists (cachefilepath) && repo.ModificationTimeTable.ContainsKey ("specs/" + file) && (DateTime) repo.ModificationTimeTable["specs/" + file] <= File.GetLastWriteTime (cachefilepath))
						{
							chosenfilepath = cachefilepath;
						}
						else
						{
							if (CachedWebDownload (repofilepath, cachefilepath))
								chosenfilepath = cachefilepath;
						}
					}
					// no caching
					else
					{
						// check if the file is avail
						if (CachedWebDownload (repofilepath, null))
							chosenfilepath = repofilepath;
					}
				}
				
				// if the file was available in one of the repos till now then stop processing
				if (chosenfilepath != null)
				{
					Repository = repo;
					break;
				}
			}
			
			// check if we were able to fetch a copy of the spec
			if (chosenfilepath != null)
			{
				XmlValidatingTextReader xvr = null;
				try
				{
					xvr = new XmlValidatingTextReader (chosenfilepath, schema);
					Load (xvr);
				}
				finally
				{
					if (xvr != null)
						xvr.Close ();
				}
			}
			//else
			//	throw (new ApplicationException ("Not able to fetch '" + file + "'; none of the available repos calaimed to have a copy of it"));
		}
		
		protected bool CachedWebDownload (string remotefile, string localfile)
		{
			if (remotefile == null)
				throw (new ArgumentNullException ("remotefile"));
			
			HttpWebRequest wc = Local.CreateHttpWebRequest (remotefile);
			
			// This way no e
			wc.AllowAutoRedirect = false;
			
			// check if the file exists locally
			if (localfile != null && File.Exists (localfile))
			{
				// Only fetch the file if it was modified on the server
				wc.IfModifiedSince = File.GetLastWriteTime (localfile);
			}
			
			// fetch the response
			try
			{
				using (HttpWebResponse wr = (HttpWebResponse)wc.GetResponse())
				{
					// check the status
					if (wr.StatusCode == HttpStatusCode.OK)
					{
						if (localfile != null)
						{
							// file was fetched so it had a newer version or was not locally available
							Stream st = wr.GetResponseStream();
							int length = (wr.ContentLength < 0 || wr.ContentLength > 8192) ? 8192 : (int)wr.ContentLength;
							byte [] buffer = new byte [length];
							using (FileStream fs = File.Create (localfile))
							{
								int nread = 0;
								while ((nread = st.Read (buffer, 0, length)) != 0)
								fs.Write (buffer, 0, nread);
							}
							// set the modified time to the same time wich the server has
							// FIXME: THIS IS A WORKAROUND MONOS WRONG IMPLEMENTATION OF HttpWebResponse.LastModified
							// (it should return local time but it returns utc)
							File.SetLastWriteTimeUtc (localfile, wr.LastModified);
						}
						
						return (true);
					}
					else if (wr.StatusCode == HttpStatusCode.NotModified)
					{
						return (true);
					}
					// throw an exception if the response status differs from the ones above
					else
					{
						throw (new ApplicationException ("Webserver returned while reqesting '" + remotefile + "': " + wr.StatusDescription));
					}
				}
			}
			// we'll get here if some deeper layered error (DNS and co) or a http error 4xx,5xx occurs
			catch (WebException e)
			{
				HttpWebResponse wr = (HttpWebResponse)e.Response;
				
				if (wr != null && wr.StatusCode == HttpStatusCode.NotFound)
				{
					if (File.Exists (localfile))
						File.Delete (localfile);
					return (false);
				}
				else
				{
					// an error occured we cant handle
					throw e;
				}
			}
		}
		
		protected static XmlSchema schema = System.Xml.Schema.XmlSchema.Read (new XmlTextReader (Config.SHAREDIR + "/upkg.xsd"), null);
	}
}
