// Applications.cs
//
// Main application for the console part of Upkg
//
// Copyright (C) 2004-2005 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.IO;
using System.Collections;
using System.Runtime.InteropServices;
using System.Net;

namespace Upkg
{
	class Application
	{
		// entry point
		static int Main (string[] args)
		{
			// Check if we have enough args
			if (args.Length == 0)
			{
				Usage();
				return 1;
			}
			
			// Parse the arguments
			Branch[] branches = Branch.DefaultBranches;
			string[] architectures = Architecture.DefaultArchitectures;
			bool withrepopath = false;
			string repopath = null;
			BranchSources bs;
			bool local = true;
			bool needed = true;
			bool missing = false;
			string packageName = null;
			string packageTag = "default";
			bool sources = true;
			bool binaries = true;
			bool states = true;
			bool specs = false;
			
			IEnumerator e = args.GetEnumerator(); 
			while (e.MoveNext())
			{
				if (((string)e.Current).StartsWith ("-"))
				{
					if (((string)e.Current).StartsWith ("--branch"))
					{
						Local.Branch = GetParamValue ((string)e.Current);
						branches = new Branch[] { Local.Branch };
					}
					else if (((string)e.Current).StartsWith ("--repos"))
					{
						Local.Repositories.Clear ();
						foreach (string repo in GetParamValue ((string)e.Current).Split (new char[] {','}))
						{
							Local.Repositories.Add (new Repository (repo));
						}
						if (Local.Repositories.Count == 0)
							throw new ApplicationException("You need to specifiy at least one valid repository.");
					}
					else if ((string)e.Current == "--help")
					{
						Usage();
						return(0);
					}
					else if ((string)e.Current == "--local")
					{
						local = true;
					}
					else if ((string)e.Current == "--remote")
					{
						local = false;
					}
					else if ((string)e.Current == "--needed")
					{
						needed = true;
					}
					else if ((string)e.Current == "--unneeded")
					{
						needed = false;
					}
					else if ((string)e.Current == "--missing")
					{
						missing = true;
					}
					else if ((string) e.Current == "--with-repo-path")
					{
						withrepopath = true;
					}
					else if (((string)e.Current).StartsWith ("--tag"))
					{
						packageTag = GetParamValue ((string) e.Current);
					}
					else if (((string)e.Current).StartsWith ("--arch"))
					{
						Local.Architecture = GetParamValue ((string)e.Current);
						Local.HostArchitecture = Local.Architecture;
						architectures = new string[] { Local.Architecture };
					}
					else if (((string)e.Current).StartsWith ("--disable-sources"))
					{
						sources = false;
					}
					else if (((string)e.Current).StartsWith ("--disable-binaries"))
					{
						binaries = false;
					}
					else if (((string)e.Current).StartsWith ("--disable-state"))
					{
						states = false;
					}
					else if (((string)e.Current).StartsWith ("--enable-specs"))
					{
						specs = true;
					}
					else if ((string) e.Current == "--debug" || (string) e.Current == "--profile" || (string) e.Current == "--trace")
					{
						// do nothing yet
					}
					else
					{
						Console.Error.WriteLine ("ERROR: Unknown option '" + (string) e.Current + "'.");
						Usage();
						return (1);
					}
				}
				else
				{
					repopath = ((string) e.Current).TrimEnd (new char[] { Path.DirectorySeparatorChar });
					if (e.MoveNext())
						packageName = (string) e.Current;
				}
			}

			if (repopath == null)
				throw (new ArgumentException ("You have to specify a path to a repository"));
			if (!needed && missing)
				throw (new ArgumentException ("Options --unneeded and --missing can't be combined."));

			Local.Verbose = false;

			if (!Directory.Exists(repopath))
				throw (new ArgumentException ("Repository does not exist"));
			// add the new repo
			Local.Repositories.Add (new Repository (repopath));

			// generate the list of needed files
			bs = new BranchSources (architectures, branches, sources, binaries, states, specs);
			if (packageName == null)
				bs.Crawl (repopath);
			else
				bs.CrawlRelease (packageName, packageTag, repopath);

			if (needed)
			{
				SortedList dict;
				
				if (local)
					dict = bs.LocalSources;
				else
					dict = bs.RemoteSources;
				
				if (withrepopath)
				{
					foreach (DictionaryEntry source in dict)
					{
						if (!missing || !File.Exists ((string) source.Key))
							Console.WriteLine ("{0}\t{1}", (string)source.Key, (string)source.Value);
					}
				}
				else
				{
					foreach (DictionaryEntry source in dict)
					{
						if (!missing || !File.Exists ((string) source.Key))
							Console.WriteLine ("{0}", (string)source.Key);
					}
				}
			}
			else
			{
				if (!local)
					throw (new ApplicationException ("Cannot list unneeded remote sources"));
				if (withrepopath)
					throw (new ApplicationException ("Cannot list unneeded sources with repo path"));
				if (!sources || !binaries || !states)
					throw (new ApplicationException ("All file types (sources, binaries, statefiles) have to be included in an unneeded file listing."));
				
				// generate a list of all real files in the repo
				DeepFileList filelist = new DeepFileList (new string[] {Path.Combine (repopath, "sources"), Path.Combine (repopath, "binaries"), Path.Combine (repopath, "state")});
				
				// compare those lists
				ListComparer lc = new ListComparer (filelist.GetKeyList(), bs.LocalSources.GetKeyList ());
				foreach (object file in lc.MissingInL2)
				{
					Console.WriteLine (file);
				}
			}

			return (0);
		}
		
		protected static string GetParamValue (string param)
		{
			string[] splitted_params = param.Split (new char[] {'='});
			
			// check if the array has correct length
			if (splitted_params.Length != 2)
				throw new ArgumentException ("Application: Invalid Parameter: " + param);
				
			return (splitted_params[1]);
		}
		
		static void Usage ()
		{
			Console.WriteLine("upkg-repo: a tool to list repository sources.");
			Console.WriteLine("Usage: upkg-repo [options] <repository path>");
			Console.WriteLine("\nOptions:");
			Console.WriteLine("  --help\t\tPrint this text.");
		}
		
		// Imported from native libc to make our script executable
		// See: "man 2 chmod" for more information about this one
		[DllImportAttribute("libc")]
		public static extern int chmod (string s, int i);
		
		protected static ArrayList releaseList;
	}
}
