// Applications.cs
//
// Main application for the console part of Upkg
//
// Copyright (C) 2004-2005 Raffaele Sandrini, Jürg Billeter
//
// This file is part of Upkg (http://www.upkg.org).
//
// Upkg is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// as published by the Free Software Foundation.
//
// Upkg is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Upkg; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Authors:
//   Raffaele Sandrini <rasa at paldo dot org>
//   Jürg Billeter <juerg at paldo dot org>

using System;
using System.IO;
using System.Collections;
using System.Runtime.InteropServices;
using System.Net;

namespace Upkg
{
	class Application
	{
		// entry point
		static int Main (string[] args)
		{
			// Check if we have enough args
			if (args.Length == 0)
			{
				Usage();
				return 1;
			}
			
			// Parse the arguments
			string outfile = null;
			string packageName = null;
			string packageTag = "default";
			bool test = false;
			IEnumerator e = args.GetEnumerator();
			bool install = false;
			bool remove = false;
			bool force = false;
			while (e.MoveNext())
			{
				if (((string)e.Current).StartsWith ("-"))
				{
					if (((string)e.Current).StartsWith ("--bootstrap"))
					{
						Local.Bootstrap = true;
						install = true;
						Local.IgnoreSelection = true;
					}
					else if (((string)e.Current).StartsWith ("--install"))
					{
						install = true;
						Local.IgnoreSelection = true;
					}
					else if (((string)e.Current).StartsWith ("--add"))
					{
						install = true;
					}
					else if (((string)e.Current).StartsWith ("--remove"))
					{
						remove = true;
					}
					else if (((string)e.Current).StartsWith ("--force"))
					{
						force = true;
						Local.IgnoreSelection = true;
					}
					else if (((string)e.Current).StartsWith ("--tag"))
					{
						packageTag = GetParamValue ((string)e.Current);
					}
					else if (((string)e.Current).StartsWith ("--branch"))
					{
						Local.Branch = GetParamValue ((string)e.Current);
					}
					else if ((string)e.Current == "--disable-source")
					{
						Local.AllowSourceBuild = false;
					}
					else if ((string)e.Current == "--enable-source")
					{
						Local.AllowSourceBuild = true;
					}
					else if ((string)e.Current == "--disable-binary")
					{
						Local.AllowBinaryInstall = false;
					}
					else if ((string)e.Current == "--enable-binary")
					{
						Local.AllowBinaryInstall = true;
					}
					else if ((string)e.Current == "--enable-caching")
					{
						Local.EnableCaching = true;
					}
					else if ((string)e.Current == "--disable-caching")
					{
						Local.EnableCaching = false;
					}
					else if (((string)e.Current).StartsWith ("--repos"))
					{
						Local.Repositories.Clear ();
						foreach (string repo in GetParamValue ((string)e.Current).Split (new char[] {','}))
						{
							Local.Repositories.Add (new Repository (repo));
						}
						if (Local.Repositories.Count == 0)
							throw new ApplicationException("You need to specifiy at least one valid repository.");
					}
					else if (((string)e.Current).StartsWith ("--arch"))
					{
						Local.Architecture = GetParamValue ((string)e.Current);
					}
					else if ((string)e.Current == "--stdout")
					{
						Local.Stdout = true;
					}
					else if ((string)e.Current == "--quiet")
					{
						Local.Verbose = false;
					}
					else if ((string)e.Current == "--test")
					{
						test = true;
						Local.EnableCaching = false;
					}
					else if ((string)e.Current == "--help")
					{
						Usage();
						return(0);
					}
					else if ((string) e.Current == "--version")
					{
						Console.WriteLine ("Upkg " + Config.VERSION);
						return (0);
					}
					else if ((string) e.Current == "--verbose")
					{
						Local.Verbose = true;
					}
					else if ((string) e.Current == "--ignore-selection")
					{
						// ignore .select files and don't remove any packages
						// used by upkg upgrade script generator
						Local.IgnoreSelection = true;
					}
					else if ((string) e.Current == "--debug" || (string) e.Current == "--profile" || (string) e.Current == "--trace")
					{
						// do nothing yet
					}
					else if (((string)e.Current).StartsWith ("--proxy"))
					{
						Local.Proxy = GetParamValue ((string)e.Current);
					}
					else
					{
						Console.Error.WriteLine ("ERROR: Unknown option '" + (string) e.Current + "'.");
						Usage();
						return (1);
					}
				}
				else
				{
					string name = (string)e.Current;
					if (e.MoveNext())
					{
						outfile = (string)e.Current;

						if ((!install && !remove) || e.MoveNext ())
						{
							Console.Error.WriteLine ("ERROR: Too many arguments.");
							Usage ();
							return (1);
						}
						packageName = name;
					}
					else
					{
						if (install || remove)
						{
							Console.Error.WriteLine ("ERROR: Too few arguments.");
							Usage ();
							return (1);
						}
						// there was no name there was only an outfile
						outfile = name;
					}
				}
			}
			if (!Local.AllowSourceBuild && !Local.AllowBinaryInstall)
			{
				Console.Error.WriteLine ("ERROR: Either binary or source installation has to be enabled.");
				Usage ();
				return (1);
			}
			
			if (force && !install)
			{
				Console.Error.WriteLine ("ERROR: --force used without --install.");
				Usage ();
				return (1);
			}
			
			// Check if we have enough info
			if (outfile == null)
			{
				Console.Error.WriteLine("ERROR: Too few arguments.");
				Usage();
				return(1);
			}

			string filepath = ((BranchSpecification)Global.Branches.Current).Settings.Variables["$CACHEDIR"] + "/" + outfile;

			// Check whether the installed upkg version is recent enough
			if (!Local.Bootstrap && Config.VERSION_URL != "" && Config.UPGRADE_SCRIPT_URL != "")
			{
				try
				{
					// get the required version number from the web
					using (WebClient wc = new WebClient ())
					{
						string minVersion = "0";
						using (StreamReader versionReader = new StreamReader (wc.OpenRead (Config.VERSION_URL)))
						{
							minVersion = versionReader.ReadLine ();
						}
						// check whether this version is too old
						if (new Version (Config.VERSION) < new Version (minVersion))
						{
							if (File.Exists (filepath))
								File.Delete (filepath);
							wc.DownloadFile (Config.UPGRADE_SCRIPT_URL, filepath);
							chmod (filepath, 0x100 | 0x80 | 0x40 | 0x20 | 0x4);
							// let the user first upgrade upkg itself
							return (0);
						}
					}
				}
				catch (Exception ex)
				{
					// ignore errors on upgrade, maybe we just don't have an internet connection at the moment
				}
			}
			
			ReleaseList selectList = new ReleaseList ();
			if (!Local.IgnoreSelection)
				selectList.AddSelectedReleases ();
			
			if (install)
			{
				try
				{
					selectList.AddRelease (packageName, packageTag);
				}
				catch (PackageNotFoundException ex)
				{
					Console.Error.WriteLine ("ERROR: " + ex.Message);
					return (1);
				}
			}
			
			bool packageFound = false;
			string packageUniqueName = ReleaseSpecification.GetUniqueName (packageName, packageTag);

			if (install && force)
			{
				releaseList = selectList;
			}
			else
			{
				releaseList = new ArrayList ();
				foreach (ReleaseSpecification rs in selectList)
				{
					if (remove && rs.UniqueName == packageUniqueName)
					{
						packageFound = true;
						continue;
					}
					rs.Selected = true;
					AddRelease (rs);
				}
				if (remove && !packageFound)
				{
					Console.Error.WriteLine ("ERROR: Package '" + packageUniqueName + "' not found in list of selected packages.");
					return (1);
				}
			}
			
			bool found_bash = false;
			
			// Write the file
			if (!test)
			{
				Console.Error.WriteLine("Writing script...");

				BuildScriptCollection bsc = null;
				// since we want to store the script relative to the cachedir we need to be sure it really exists
				if (!Directory.Exists((string) ((BranchSpecification)Global.Branches.Current).Settings.Variables["$CACHEDIR"]))
					Directory.CreateDirectory((string) ((BranchSpecification)Global.Branches.Current).Settings.Variables["$CACHEDIR"]);
				// Create the writer
				if (!Local.Stdout)
				{
					try
					{
						bsc = new BuildScriptCollection (filepath);
					}
					catch (UnauthorizedAccessException ex)
					{
						Console.Error.WriteLine ("ERROR: Write access to the path '" + filepath + "' is denied.");
						return (1);
					}
				}
				else
				{
					bsc = new BuildScriptCollection (Console.OpenStandardOutput (), filepath);
				}

				foreach (ReleaseSpecification release in releaseList) {
					if (release.UniqueName == "bash") {
						found_bash = true;
					}
					bsc.AddReleaseSpecification (release);
				}

				bsc.Write ();
				chmod (filepath, 0x100 | 0x80 | 0x40 | 0x20 | 0x4);
			}
			
			if (!found_bash && !Local.IgnoreSelection) {
				// don't destroy system
				Console.Error.WriteLine ("ERROR: Invalid package selection, cowardly refusing to remove bash.");
				return (1);
			}
			
			// We get here? its all done then.
			return(0);
		}
		
		protected static void AddRelease (ReleaseSpecification release)
		{
			if (release == null)
				throw (new ArgumentNullException ("release"));
		
			// Don't add package if we're not bootstrapping and package is not installable
			if (!release.Zombie && !Local.Bootstrap && release.Settings.Staging)
				return;

			if (releaseList.Contains (release))
				return;
			
			foreach (ReleaseSpecification dep in release.Deps)
				AddRelease (dep);
			
			releaseList.Add (release);
		}		
		
		protected static string GetParamValue (string param)
		{
			string[] splitted_params = param.Split (new char[] {'='});
			
			// check if the array has correct length
			if (splitted_params.Length != 2)
				throw new ArgumentException ("Application: Invalid Parameter: " + param);
				
			return (splitted_params[1]);
		}
		
		static void Usage ()
		{
			Console.WriteLine("upkg: a tool to compile Upkg XML specs into bash scripts.");
			Console.WriteLine("Usage: upkg [options] [package] <output file>");
			Console.WriteLine("\nOptions:");
			Console.WriteLine("  --install\t\tInstall the specified package.");
			Console.WriteLine("  --remove\t\tRemove the specified package.");
			Console.WriteLine("  --bootstrap\t\tBuild the specified package in a separate root dir.");
			Console.WriteLine("  --disable-source\tOnly install packages available as binaries.");
			Console.WriteLine("  --disable-binary\tDo not use binary packages. Compile them instead.");
			Console.WriteLine("  --disable-caching\tDo not cache XML specs.");
			Console.WriteLine("  --stdout\t\tDo not write to output file. Use standard output.");
			Console.WriteLine("  --repos=<repo>,...\tUse specified repositories.");
			Console.WriteLine("  --tag=<name>\t\tUse tag '<name>' within [package].");
			Console.WriteLine("  --branch=<name>\tThe branch which will be used.");
			Console.WriteLine("  --proxy=<url>\t\tThe proxy which will be used.");
			Console.WriteLine("  --test\t\tPerforms a dry-run. Doesn't write anything.");
			Console.WriteLine("  --help\t\tPrint this text.");
			Console.WriteLine("\nATTENTION: <output file> will always be relative to your upkg cache directory!");
		}
		
		// Imported from native libc to make our script executable
		// See: "man 2 chmod" for more information about this one
		[DllImportAttribute("libc")]
		public static extern int chmod (string s, int i);
		
		protected static ArrayList releaseList;
	}
}
