/*
   upkg-watch.c - tool to log file modifications

   Copyright (C) 2004 Juerg Billeter
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; version 2 of the
   License.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
   
   This is based on installwatch written by Pancrazio 'Ezio' de Mauro <p@demauro.net>

   Author: Juerg Billeter <j@bitron.ch>
*/

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <dlfcn.h>
#include <syslog.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <stddef.h>

static int(*true_creat)(const char *, mode_t);
static FILE *(*true_fopen)(const char *,const char*);
static int(*true_link)(const char *, const char *);
static int(*true_mkdir)(const char *, __mode_t);
static int(*true_open)(const char *, int, ...);
static int(*true_rename)(const char *, const char *);
static int(*true_symlink)(const char *, const char *);
static int(*true_truncate)(const char *, off_t);
static int(*true_creat64)(const char *, __mode_t);
static FILE *(*true_fopen64)(const char *,const char *);
static int(*true_open64)(const char *, int, ...);
static int(*true_truncate64)(const char *, __off64_t);
static int(*true_linkat)(int, const char *, int, const char *, int);
static int(*true_mkdirat)(int, const char *, __mode_t);
static int(*true_openat)(int, const char *, int, ...);
static int(*true_renameat)(int, const char *, int, const char *);
static int(*true_renameat2)(int, const char *, int, const char *, unsigned int);
static int(*true_symlinkat)(const char *, int, const char *);
static int(*true_openat64)(int, const char *, int, ...);

static int initialized = 0;

static void doinit () __attribute__ ((constructor));

static void doinit ()
{
	void *libc_handle;

	if (initialized)
		return;

	libc_handle = RTLD_NEXT;
	true_creat = dlsym(libc_handle, "creat");
	true_fopen = dlsym(libc_handle, "fopen");
	true_link = dlsym(libc_handle, "link");
	true_mkdir = dlsym(libc_handle, "mkdir");
	true_open = dlsym(libc_handle, "open");
	true_rename = dlsym(libc_handle, "rename");
	true_symlink = dlsym(libc_handle, "symlink");
	true_truncate = dlsym(libc_handle, "truncate");
	true_creat64 = dlsym(libc_handle, "creat64");
	true_fopen64 = dlsym(libc_handle, "fopen64");
	true_open64 = dlsym(libc_handle, "open64");
	true_truncate64 = dlsym(libc_handle, "truncate64");
	true_linkat = dlsym(libc_handle, "linkat");
	true_mkdirat = dlsym(libc_handle, "mkdirat");
	true_openat = dlsym(libc_handle, "openat");
	true_renameat = dlsym(libc_handle, "renameat");
	true_renameat2 = dlsym(libc_handle, "renameat2");
	true_symlinkat = dlsym(libc_handle, "symlinkat");
	true_openat64 = dlsym(libc_handle, "openat64");
	initialized = 1;
}

char *
realpath_nofollow (const char *name, char *resolved)
{
  char *rpath, *dest = NULL;
  const char *start, *end, *rpath_limit;
  long int path_max;

  if (name == NULL)
    {
      return NULL;
    }

  if (name[0] == '\0')
    {
      return NULL;
    }

#ifdef PATH_MAX
  path_max = PATH_MAX;
#else
  path_max = pathconf (name, _PC_PATH_MAX);
  if (path_max <= 0)
    path_max = 1024;
#endif

  if (resolved == NULL)
    {
      rpath = malloc (path_max);
      if (rpath == NULL)
	return NULL;
    }
  else
    rpath = resolved;
  rpath_limit = rpath + path_max;

  if (name[0] != '/')
    {
      if (!getcwd (rpath, path_max))
	{
	  rpath[0] = '\0';
	  goto error;
	}
      dest = strchr (rpath, '\0');
    }
  else
    {
      rpath[0] = '/';
      dest = rpath + 1;
    }

  for (start = end = name; *start; start = end)
    {
      struct stat64 st;

      /* Skip sequence of multiple path-separators.  */
      while (*start == '/')
	++start;

      /* Find end of path component.  */
      for (end = start; *end && *end != '/'; ++end)
	/* Nothing.  */;

      if (end - start == 0)
	break;
      else if (end - start == 1 && start[0] == '.')
	/* nothing */;
      else if (end - start == 2 && start[0] == '.' && start[1] == '.')
	{
	  /* Back up to previous component, ignore if at root already.  */
	  if (dest > rpath + 1)
	    while ((--dest)[-1] != '/');
	}
      else
	{
	  size_t new_size;

	  if (dest[-1] != '/')
	    *dest++ = '/';

	  if (dest + (end - start) >= rpath_limit)
	    {
	      ptrdiff_t dest_offset = dest - rpath;
	      char *new_rpath;

	      if (resolved)
		{
		  if (dest > rpath + 1)
		    dest--;
		  *dest = '\0';
		  goto error;
		}
	      new_size = rpath_limit - rpath;
	      if (end - start + 1 > path_max)
		new_size += end - start + 1;
	      else
		new_size += path_max;
	      new_rpath = (char *) realloc (rpath, new_size);
	      if (new_rpath == NULL)
		goto error;
	      rpath = new_rpath;
	      rpath_limit = rpath + new_size;

	      dest = rpath + dest_offset;
	    }

	  dest = __mempcpy (dest, start, end - start);
	  *dest = '\0';

	  if (lstat64 (rpath, &st) < 0)
	    goto error;

	}
    }
  if (dest > rpath + 1 && dest[-1] == '/')
    --dest;
  *dest = '\0';

  return resolved ? memcpy (resolved, rpath, dest - rpath + 1) : rpath;

error:
  if (resolved)
    strcpy (resolved, rpath);
  else
    free (rpath);
  return NULL;
}

static void log (const char *pathname)
{
	int true_errno;
	char canonic[MAXPATHLEN];

	true_errno = errno;

	if (!initialized)
		doinit ();

	realpath_nofollow (pathname, canonic);
	
	if (strncmp (canonic, "/dev/", 5) && strncmp (canonic, "/proc/", 6) && strncmp (canonic, "/sys/", 5) && strncmp (canonic, "/tmp/", 5) && strncmp (canonic, "/src/", 5))
	{
		FILE *logfile = true_fopen ("/tmp/upkg-watch.log", "a");
		fprintf (logfile, "%s\n", canonic);
		fclose (logfile);
	}

	errno = true_errno;
}

static void logat (int dirfd, const char *pathname)
{
	int true_errno;
	char proc_path[32];
	char resolved_path[MAXPATHLEN];
	ssize_t length;

	if (dirfd == AT_FDCWD)
	{
		log (pathname);
		return;
	}

	true_errno = errno;

	snprintf (proc_path, 32, "/proc/self/fd/%d", dirfd);
	length = readlink (proc_path, resolved_path, MAXPATHLEN);

	resolved_path[length] = '/';
	strncpy (resolved_path + length + 1, pathname, MAXPATHLEN - length - 1);
	resolved_path[MAXPATHLEN - 1] = '\0';

	log (resolved_path);

	errno = true_errno;
}

static void dirs_log (const char *pathname)
{
	int true_errno;
	char canonic[MAXPATHLEN];

	true_errno = errno;

	if (!initialized)
		doinit ();

	realpath_nofollow (pathname, canonic);
	
	if (strncmp (canonic, "/dev/", 5) && strncmp (canonic, "/proc/", 6) && strncmp (canonic, "/sys/", 5) && strncmp (canonic, "/tmp/", 5) && strncmp (canonic, "/src/", 5))
	{
		FILE *logfile = true_fopen ("/tmp/upkg-watch.dirs.log", "a");
		fprintf (logfile, "%s\n", canonic);
		fclose (logfile);
	}

	errno = true_errno;
}

static void dirs_logat (int dirfd, const char *pathname)
{
	int true_errno;
	char proc_path[32];
	char resolved_path[MAXPATHLEN];
	ssize_t length;

	if (dirfd == AT_FDCWD)
	{
		dirs_log (pathname);
		return;
	}

	true_errno = errno;

	snprintf (proc_path, 32, "/proc/self/fd/%d", dirfd);
	length = readlink (proc_path, resolved_path, MAXPATHLEN);

	resolved_path[length] = '/';
	strncpy (resolved_path + length + 1, pathname, MAXPATHLEN - length - 1);
	resolved_path[MAXPATHLEN - 1] = '\0';

	dirs_log (resolved_path);

	errno = true_errno;
}

int creat (const char *pathname, mode_t mode)
{
	int result;
	                                                                        
	if (!initialized)
		doinit ();

	result = true_creat (pathname, mode);

	log (pathname);
	
	return result;
}

FILE *fopen (const char *pathname, const char *mode)
{
	FILE *result;
	
	if (!initialized)
		doinit ();

	result = true_fopen (pathname, mode);
	
	if (mode[0] == 'w' || mode[0] == 'a' || mode[1] == '+')
		log (pathname);

	return result;
}

int link (const char *oldpath, const char *newpath)
{
	int result;
	
	if (!initialized)
		doinit ();

	result = true_link (oldpath, newpath);

	log (newpath);

	return result;
}

int mkdir (const char *pathname, mode_t mode)
{
	int result;
	
	if (!initialized)
		doinit ();

	result = true_mkdir (pathname, mode);

	dirs_log (pathname);

	return result;
}

int open (const char *pathname, int flags, ...)
{
	/* there may be a third parameter: mode_t mode */
	va_list ap;
	mode_t mode;
	int result;
	
	if (!initialized)
		doinit ();

	va_start(ap, flags);
	mode = va_arg (ap, mode_t);
	va_end(ap);
	
	result = true_open (pathname, flags, mode);
	if (flags & (O_WRONLY | O_RDWR))
		log (pathname);

	return result;
}

int rename(const char *oldpath, const char *newpath) {
	int result;
	
	if (!initialized)
		doinit ();

	result = true_rename (oldpath, newpath);

	log (newpath);
	
	return result;
}

int symlink (const char *oldpath, const char *newpath)
{
	int result;
	                                                                        
	if (!initialized)
		doinit ();

	result = true_symlink (oldpath, newpath);

	log (newpath);

	return result;
}

int truncate (const char *path, off_t length)
{
        int result;
                                                                                
	if (!initialized)
		doinit ();

        result = true_truncate(path, length);

	log (path);

        return result;
}

int creat64 (const char *pathname, __mode_t mode)
{
	int result;

	if (!initialized)
		doinit ();

	result = true_creat64 (pathname, mode);

	log (pathname);

	return result;
}

FILE *fopen64 (const char *pathname, const char *mode)
{
	FILE *result;
	                                                                
	if (!initialized)
		doinit ();

	result = true_fopen64 (pathname, mode);

	if (mode[0] == 'w' || mode[0] == 'a' || mode[1] == '+')
		log (pathname);

	return result;
}

int open64 (const char *pathname, int flags, ...)
{
	/* there may be a third parameter: mode_t mode */

	va_list ap;
	mode_t mode;
	int result;
	
	va_start(ap, flags);
	mode = va_arg (ap, mode_t);
	va_end(ap);
	
	result = true_open64 (pathname, flags, mode);
	if (flags & (O_WRONLY | O_RDWR))
		log (pathname);

	return result;
}

int truncate64 (const char *path, __off64_t length)
{
        int result;
                                                                                
	if (!initialized)
		doinit ();

        result = true_truncate64 (path, length);

	log (path);

        return result;
}

int linkat (int olddirfd, const char *oldpath, int newdirfd, const char *newpath, int flags)
{
	int result;

	if (!initialized)
		doinit ();

	result = true_linkat (olddirfd, oldpath, newdirfd, newpath, flags);

	logat (newdirfd, newpath);

	return result;
}

int mkdirat (int dirfd, const char *pathname, mode_t mode)
{
	int result;

	if (!initialized)
		doinit ();

	result = true_mkdirat (dirfd, pathname, mode);

	dirs_logat (dirfd, pathname);

	return result;
}

int openat (int dirfd, const char *pathname, int flags, ...)
{
	/* there may be a fourth parameter: mode_t mode */

	va_list ap;
	mode_t mode;
	int result;

	if (!initialized)
		doinit ();

	va_start(ap, flags);
	mode = va_arg (ap, mode_t);
	va_end(ap);

	result = true_openat (dirfd, pathname, flags, mode);
	if (flags & (O_WRONLY | O_RDWR))
		logat (dirfd, pathname);

	return result;
}

int renameat (int olddirfd, const char *oldpath, int newdirfd, const char *newpath)
{
	int result;

	if (!initialized)
		doinit ();

	result = true_renameat (olddirfd, oldpath, newdirfd, newpath);

	logat (newdirfd, newpath);

	return result;
}

int renameat2 (int olddirfd, const char *oldpath, int newdirfd, const char *newpath, unsigned int flags)
{
	int result;

	if (!initialized)
		doinit ();

	result = true_renameat2 (olddirfd, oldpath, newdirfd, newpath, flags);

	logat (newdirfd, newpath);

	return result;
}

int symlinkat (const char *oldpath, int newdirfd, const char *newpath)
{
	int result;

	if (!initialized)
		doinit ();

	result = true_symlinkat (oldpath, newdirfd, newpath);

	logat (newdirfd, newpath);

	return result;
}

int openat64 (int dirfd, const char *pathname, int flags, ...)
{
	/* there may be a fourth parameter: mode_t mode */

	va_list ap;
	mode_t mode;
	int result;

	va_start(ap, flags);
	mode = va_arg (ap, mode_t);
	va_end(ap);

	result = true_openat64 (dirfd, pathname, flags, mode);
	if (flags & (O_WRONLY | O_RDWR))
		logat (dirfd, pathname);

	return result;
}

